#pragma once

#ifndef ML_BERTAPI_H_
#define ML_BERTAPI_H_

#ifndef APICALL
#ifdef __cplusplus
#   ifdef WIN32
#       ifndef ML_IMPORTS_API
#           define APICALL extern "C" __declspec(dllexport)
#       else
#           define APICALL extern "C" __declspec(dllimport)
#       endif
#   else
#       define APICALL extern "C" 
#   endif // WIN32
#else
#   define APICALL
#endif
#endif

#ifdef WINAPI
#define STACKMODE   WINAPI
#else
#ifndef WIN32
#define __stdcall
#endif
#define STACKMODE   __stdcall
#endif


#define MLADDRLEN       256     // Maximum address length


enum MLCONNTYPE
{
	UNSUPPORTED = -1,   // Unsupported type (usually used in return or default type)
	ETHERNET = 0,       // Ethernet connection
	FPGA = 1,           // FPGA connection
	PXI = 2,            // PCIe FPGA connection
};



struct mlBertConnectionParams
{
	char address[MLADDRLEN];    // Connection address (IP)
	MLCONNTYPE type;              // Connection protocol type
#ifdef ML_DSO_API
	bool isOptical;             // Flag indicating optical mode
	int opticalWavelength;      // Optical wavelength
	bool isSimulated;           // Flag indicating simulation mode
#endif
};



enum MLEyeMode_ML4004 {
	mlPam4 = 0,				// PAM4 modulation
	mlNRZ					// NRZ modulation
};

enum MLUserDefinedFileFormat
{
	mlBinary = 0,
	mlPAM,
	mlHEX
};


// Types
typedef     unsigned char       byte;
typedef     byte                UInt8;
typedef     unsigned short      UInt16;
typedef     unsigned short      USHORT;
typedef     unsigned long       UInt32;
typedef     unsigned long long  UInt64;

typedef		unsigned short      UINT16, *PUINT16;

struct  mlbertapi;


//++
//
//  mlbertapi* STACKMODE mlBert_createInstance();
//
//  Description:
//      Create a new BERT API instance.
//      It is recommended that each device should have its own instance.
//
//  Returns:
//      Pointer to created BERT API instance.
//
//--
APICALL mlbertapi* STACKMODE mlBert_createInstance();

////////////////////////////////////////////////////////////////////////////////////////////
// Connection Section
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_Connect(mlbertapi* instance, std::string ip)
//
// Routine Description:
//
//    Connect to Board
//
// Arguments:
//
//    ip: The IP of the board to be connected.
//
// Return Value:
//
//    Returns if the connection was successfully made or not
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Connect(mlbertapi* instance, char* IP);
APICALL int  STACKMODE mlBert_Connection(mlbertapi* instance, mlBertConnectionParams conn);
#pragma region declaration
//++
//
//	int mlBert_Disconnect(mlbertapi* instance)
//
// Routine Description:
//
//    Disconnect from the board
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if disconnecting from the board was successfully executed
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Disconnect(mlbertapi* instance);

////////////////////////////////////////////////////////////////////////////////////////////
// Board Settings
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_ReadBoardID(mlbertapi* instance)
//
// Routine Description:
//
//    Read Board ID to Identify board Type.
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns Board ID.
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadBoardID(mlbertapi* instance);
#pragma region declaration
//++
//
//	int mlBert_ReadBoardSN(mlbertapi* instance)
//
// Routine Description:
//
//    Read Board SN.
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns Board SN.
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadBoardSN(mlbertapi* instance);
#pragma region declaration
//++
//
//	int mlBert_ReadSerialNumber(int * SerialNumberVersion)
//
// Routine Description:
//
//    Read Board SN.
//
// Arguments:
//    instance :API instance
//    SN  :the serial number;
//    SerialNumberVersion = 0 represetn old serial number system, 1 represent the new SN version 
//
// Return Value:
//
//    Returns Board SN.
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadSerialNumber(mlbertapi* instance, UInt64 * SN, int *SerialNumberVersion);
#pragma region declaration
//++
//
//	int mlBert_WriteBoardSN(mlbertapi* instance, int boardSN)
//
// Routine Description:
//
//    Write Board SN.
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_WriteBoardSN(mlbertapi* instance, int boardSN);
#pragma region declaration
//++
//
//	double mlBert_ReadFirmwareRevision(mlbertapi* instance)
//
// Routine Description:
//
//    Read Firmware Revision loaded in the board.
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns Firmware Revision.
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_ReadFirmwareRevision(mlbertapi* instance);

////////////////////////////////////////////////////////////////////////////////////////////
// Application Configuration
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_ConfigureApplication(
//            mlbertapi* instance, char* saveConfig,
//            char* saveBathtub, char* saveEye,
//            int saveBathtubEnable, int saveEyeEnable)
//
// Routine Description:
//
//    Configure Application parameters like
//    clock location files, bathtub curve and eye save location.
//
// Arguments:
//
//    instance :API instance
//    saveConfig: Location of the Clock File generated by 4x25G-ClockCalculation software or ClockGeneratorAPI.
//    saveBathtub: Save location of the Bathtub Curve files.
//    saveEye: Save location of the Eye files.
//    saveBathtubEnable: Enable = 1 or Disable = 0 Bathtub Saving.
//    saveEyeEnable: Enable = 1 or Disable = 0 Eye Saving
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_ConfigureApplication(mlbertapi* instance, char* saveConfig, char* saveBathtub, char* saveEye, int saveBathtubEnable, int saveEyeEnable);
#pragma region declaration
//++
//
//	int mlBert_AccessBoardRegister(mlbertapi* instance, byte Read_Write, UInt16 Index,  UInt16 Reg,  UInt16* Data)
//
// Routine Description:
//
//    Read Board ID to Identify board Type.
//
// Arguments:
//
//    instance :API instance
//    Read_Write: 0 = Read, 1 = Write
//    Index: Index of the chip
//    Reg: Reg of the ship
//    Data: Returned data in Read operation and Saved data in Write Operation
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_AccessBoardRegister(mlbertapi* instance, byte Read_Write, UInt16 Index, UInt16 Reg, UInt16* Data);
#pragma region declaration
//++
//
//	int mlBert_RestoreAllConfig()
//
// Routine Description:
//
//    Restore all configurations
//    used to restore configuration after power cycle or after applying linerate
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_RestoreAllConfig(mlbertapi* instance);
#pragma region declaration
//++
//
//	int mlBert_ClockOut(mlbertapi* instance, int clockIndex)
//
// Routine Description:
//
//    Specify clock out
//
// Arguments:
//
//    instance :API instance
//    channel   : CDR Channel Select
//    clockIndex: 0 = Internal(Linerate>21 = Linerate/80 else Linerate<=15 = Linerate/32)
//                1 = CDR8 Linerate/8
//                2 = CDR16 Linerate/16
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOut(mlbertapi* instance, int channel, int clockIndex);

APICALL int  STACKMODE mlBert_RateClock(mlbertapi* instance, int clockIndex);

#pragma region declaration
//++
//
//	int mlBert_ClockOut3b(mlbertapi* instance, int clockIndex, int ReferenceClockOutRate)
//
// Routine Description:
//
//    Specify clock out for Ml403B
//
// Arguments:
//
//    instance :API instance
//    clockIndex: 0 = Silab reference clock
//                1 = Avago TX Clock
//                2 = Monitor Clock
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_ClockOut3b(mlbertapi* instance, int clockIndex, int ReferenceClockOutRate);

#pragma region declaration
//++
//
//	int mlBert_ClockOut4439(int channel ,int RateOver)
//
// Routine Description:
//
//    Specify clock out for ML4039b eql
//
// Arguments:
//
//    channel   : 0 = channle 1 output
//                1 = channle 2 output
//                2 = channle 3 output
//                3 = channle 4 output
//    RateOver    0 = representing reference clock out
//                1 = representing rate/80
//                2 = representing rate/40
//                3 = representing rate/20
//                4 = representing rate/10
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOut4439(mlbertapi* instance, int channel, int RateOver);

#pragma region declaration
//++
//
//	int mlBert_ClockOut39A(int channel)
//
// Routine Description:
//
//    Specify clock out for ML4039A jit 
//
// Arguments:
//
//    instance  : API instance
//    TriggerOut: 0 = refernce clock
//                1 = refernce clock/2
//                2 = CDR 
//    CDRchannel: 0 = CDR channel 1 output (if TriggerOut = 2 channel 0 from clean chip and if 1 channel 0 from jit chip)
//                1 = CDR channel 2 output (if TriggerOut = 2 channel 1 from clean chip and if 1 channel 1 from jit chip)
//                2 = CDR channel 3 output (if TriggerOut = 2 channel 2 from clean chip and if 1 channel 2 from jit chip)
//                3 = CDR channel 4 output (if TriggerOut = 2 channel 3 from clean chip and if 1 channel 3 from jit chip)
//    CDRRate   : 0 = Datarate/8
//                1 = Datarate/16
//                2 = Datarate/32
//                3 = Datarate/64
//                4 = Datarate/128
//                5 = Datarate/256

// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOut39A(mlbertapi* instance, int TriggerOut, int CDRchannel, int CDRRate);
#pragma region declaration
//++
//
//	int  mlBert_ClockOut4054(mlbertapi* instance, int TriggerOut, int MonitorSetting, int CDRchannel, int CDRRate)
//
// Routine Description:
//
//    Specify clock out for ML4054 
//
// Arguments:
//
//    instance  : API instance
//    TriggerOut: 0 = refernce clock QSFP1
//                1 = refernce clock QSFP2
//                2 = Monitor 1
//                3 = Monitor 2
//                
//    MonitorSetting : 0 = LPCDR A
//                     1 = TX PLL
//                     2= CDR
//
//    CDRchannel: CDR channel or TX clock Pll channel form 0 to 7
//
//    CDRRate   : 0 = Datarate/8
//                1 = Datarate/16
//                2 = Datarate/32
//                3 = Datarate/64
//                4 = Datarate/128
//                5 = Datarate/256
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOut4054(mlbertapi* instance, int TriggerOut, int MonitorSetting, int CDRchannel, int CDRRate);
#pragma region declaration
//++
//
//	int  mlBert_ClockOut4644(mlbertapi* instance, int TriggerOut, int CDRRate)
//
// Routine Description:
//
//    Specify clock out for ML4039B 
//
// Arguments:
//
//    instance  : API instance
//    TriggerOut: 0 = Monitor
//                1 = External
//                2 = Reference clk
//                
//    CDRRate   : 32
//                64
//                128
//                256
//                512
//                1024
//				  2048
//				  4096
// Return Value:  
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOut4644(mlbertapi* instance, int TriggerOut, int CDRRate);

//    instance  : API instance
//    TriggerOut: 0 = refernce clock QSFPA
//                1 = refernce clock QSFPC
//                2 = Monitor GB2 A
//                3 = Monitor GB2 C

APICALL int  STACKMODE mlBert_ClockOut4070(mlbertapi* instance, int TriggerOut, int CDRchannel, int Rate);

#pragma region declaration
//++
//
//	int mlBert_ClockOutTDArevB(mlbertapi* instance, int ClockInSource, int ClockOutSource, int ClockOutDividers);
//
// Routine Description:
//
//    Specify clock Intput and output for TDA rev B
//
// Arguments:
//
//    instance  : API instance
//    ClockInSource: 0 = InputClock silab source
//                   1 = InputClock FM Circuit
//                
//    ClockOutSource : 0 = LPCDR 
//                     1 = LPCDR TX Clock (LineRate/ ClockOutDividers) 
//                     2 = LPCDR RX Clock (LineRate/ ClockOutDividers) 
//
//    ClockOutDividers : 0 = Datarate/8
//                       1 = Datarate/16
//                       2 = Datarate/32
//                       3 = Datarate/64
//                       4 = Datarate/128
//                       5 = Datarate/256
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ClockOutTDArevB(mlbertapi* instance, int ClockInSource, int ClockOutSource, int ClockOutDividers);

#pragma region declaration
//++
//
//	int mlBert_TXClockOut_RateOverEight(mlbertapi* instance)
//
// Routine Description:
//
//    Specify clock out equal to line rate over eight for ML4039, ML4039jit, ML4009, ML4009jit, ML4009eql and ML4039A 
//
// Arguments:
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_TXClockOut_RateOverEight(mlbertapi* instance);
#pragma region declaration
//++
//
//	int mlBert_RestoreAllConfig(mlbertapi* instance)
//
// Routine Description:
//
//    Restore all Configuration to last applied
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_RestoreAllConfig(mlbertapi* instance);

////////////////////////////////////////////////////////////////////////////////////////////
// TX Configuration
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_ReadAvogoDividerCalibration(mlbertapi* instance, int Channel, double LineRate)
//
// Routine Description:
//
//	Read The Divider used to configure the Avogo Chip for a specific line Rate and channel (used for ML4439 and Ml4839) 
//
// Arguments:
//
//  Instance :API instance
//  Channel: Channel Number
//	LineRate: LineRate
//
// Return Value:
//
//    Returns the divider value
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadAvogoDividerCalibration(mlbertapi* instance, int Channel, double LineRate);

////////////////////////////////////////////////////////////////////////////////////////////
// TX Configuration
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_CheckBertJitisATE(mlbertapi* instance)
//
// Routine Description:
//
//	check if the ML4039 jit is a ATE Board 
//
// Arguments:
//
//  Instance :API instance
//
// Return Value:
//
//    Returns 1 if true and 0 if false
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_CheckBertJitisATE(mlbertapi* instance);
#pragma region declaration
//++
//
//	int mlBert_LineRateConfiguration(mlbertapi* instance, double lineRate, int clockIndex)
//
// Routine Description:
//
//	Configure Board Line rate
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate required in Gbps ranging from 8.5-15 and from 21-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//	clockIndex: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration(mlbertapi* instance, double lineRate, int clockIndex);
#pragma region declaration
//++
//
//	intmlBert_LineRateConfiguration39EVB(mlbertapi* instance, double linerate, int clockSource)
//
// Routine Description:
//
//	Configure Board Line rate for EVB boards revA and revB
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate required in Gbps ranging from 8.5-15 and from 21-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//	clockIndex: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration39EVB(mlbertapi* instance, double linerate, int clockSource);

#pragma region declaration
//++
//
//	int mlBert_LineRateConfigurationBert3B(mlbertapi* instance, double linerate, int clockSource)
//
// Routine Description:
//
//	Configure Board Line rate for 4003B/4003BX
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate required in Gbps 
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//	clockIndex: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfigurationBert3B(mlbertapi* instance, double linerate, int clockSource);
#pragma region declaration
//++
//
//	int mlBert_TDALineRateConfigurationBert3B(mlbertapi* instance, double linerate, int clockSource, double *Fin_Bert, double *Fout_Bert, double *Fin_DSO, double *Fout_DSO)
//
// Routine Description:
//
//	Configure Board Line rate for 4003B with DSO
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate required in Gbps 
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//	clockIndex: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_TDALineRateConfigurationBert3B(mlbertapi* instance, double linerate, int clockSource, double *Fin_Bert, double *Fout_Bert, double *Fin_DSO, double *Fout_DSO);
#pragma region declaration
//++
//
//	int mlBert_SaveLineRatetoFW(mlbertapi* instance, int channelIndex, double linerate)
//
// Routine Description:
//
//	Save linerate to FW
//
// Arguments:
//
//  instance :API instance
//	channelIndex
//  lineRate: Contain the line rate required in Gbps 
//			
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL void  STACKMODE mlBert_SaveLineRatetoFW(mlbertapi* instance, int channelIndex, double linerate);

#pragma region declaration
//++
//
//	int mlBert_LineRateConfiguration_VCOCalibration(mlbertapi* instance, double linerate, int PhyIndex)
//
// Routine Description:
//
//	Configure Board Line rate for ML4319 it's used only for ML4319 VCO calobration Gui
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate required in Gbps 
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//	PhyIndex: Select the inphy source chip according to channel (0 to 8) 
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_VCOCalibration(mlbertapi* instance, double linerate, int PhyIndex);
#pragma region declaration
//++
//
//	int mlBert_LineRateConfiguration_04(mlbertapi* instance, double lineRate, int EyeMode, int GrayMapping, int PRECoding, int BER_FECMode, int clockSource, int FECOperationMode, int HostSide, int HostLineDebug)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4004
//
// Arguments:
//  
//  Instance :API instance
//  lineRate: Contain the line rate required in Gbps ranging from 8.5-15 and from 21-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//  EyeMode : 0 for PAM4, 1 for NRZ
//  FECMode : 0 for _802_3bj, 1 for KP4
//  RateSelect : 1 for 4:10 G and 0 for 4:25 G
//  GrayMapping : 1 to enable 0 to disable
//  PRECoding   : 1 to enable 0 to disable
//  BER_FECMode : 1 to eanble FEC Ber Mode, only used in Pam4 Ber calibration
//	ClockSource: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
//  FECOperationMode : 0 RP4, 1 KP4
//	HostSide: 1 for host side, 0 for line side
//	HostLineDebug: host and line side
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_LineRateConfiguration_04(mlbertapi* instance, double lineRate, int EyeMode, int GrayMapping, int PRECoding, int BER_FECMode, int clockSource, int FECOperationMode, int HostSide, int HostLineDebug);
#pragma region declaration
//++
//
//	int  mlBert_LineRateConfigurationBert39B(mlbertapi* instance, int channelIndex, double linerate1, double linerate2, int clockSource, int AvagoMultilplier_Ch1, int AvagoMultilplier_Ch2)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4039B-EQL
//
// Arguments:
//
//  instance :API instance
//  channelIndex: if 0 the linerate1 and linerate2 refers to channel1 and channel2, if 1 they refers to channel3, channel4
//  lineRate1: Contain the line rate(for channel 1 if channelindex = 0, channel 3 if channelindex = 1)
//             required in Gbps ranging from 1-15 and from 15-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//
//  lineRate2: Contain the line rate(for channel 2 if channelindex = 0, channel 4 if channelindex = 1)
//             required in Gbps ranging from 1-15 and from 15-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate.
//                  Configuration can be restored automatically using RestoreAllConfig() function
//
//	ClockSource1: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfigurationBert39B(mlbertapi* instance, int channelIndex, double linerate1, double linerate2, int clockSource, int AvagoMultilplier_Ch1, int AvagoMultilplier_Ch2);
#pragma region declaration
//++
//
//	int mlBert_LineRateConfigurationBert39Brv2(mlbertapi* instance, int channelIndex, double linerate1, double linerate2, int clockSource, int clockSource_ConfigIndex)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4039Brv2 (4439)
//
// Arguments:
//
//  instance :API instance
//  channelIndex: if 0 the linerate1 and linerate2 refers to channel1 and channel2, if 1 they refers to channel3, channel4
//  lineRate1: Contain the line rate(for channel 1 if channelindex = 0, channel 3 if channelindex = 1)
//             required in Gbps ranging from 1-15 and from 15-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//
//  lineRate2: Contain the line rate(for channel 2 if channelindex = 0, channel 4 if channelindex = 1)
//             required in Gbps ranging from 1-15 and from 15-28.9
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//
//	ClockSource1: Select source if internal or external Default(0)
//			  External = 0, Internal = 1
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfigurationBert39Brv2(mlbertapi* instance, int channelIndex, double linerate1, double linerate2, int clockSource, int clockSource_ConfigIndex);
#pragma region declaration
//++
//
//	int mlBert_LineRateConfiguration_4070(mlbertapi* instance, double lineRate)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4070
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate(for channel 0 to 3)
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_4070(mlbertapi* instance, double lineRate);
#pragma region declaration
//++
//
//	int  mlBert_LineRateConfiguration_Vega(mlbertapi* instance, double linerate, int clockSource, MLEyeMode_ML4004 EyeMode, int PRECoding, int GrayMapping, int FECEnable, int FECType, int side, int Taps)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4039B
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate
//	clockSource: 1 for internal, 0 for external
//	EyeMode: 0 for PAM mode,1 for NRZ mode
//	PRECoding : 1 to enable, 0 to disable
//	GrayMapping: 1 to enable, 0 to disable
//	FECEnable: 1 to enable, 0 to disable
//  FECType 
//  side: 0
//  Taps: 1 to enable, 0 to disable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_Vega(mlbertapi* instance, double linerate, int clockSource, MLEyeMode_ML4004 EyeMode, int PRECoding, int GrayMapping, int FECEnable, int FECType, int side, int Taps);
#pragma region declaration
//++
//
//	int mlBert_LineRateConfiguration_4054(mlbertapi* instance, double LineRate1, double LineRate2)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4054
//
// Arguments:
//
//  instance :API instance
//  lineRate1: Contain the line rate(for channel 0 to 3)
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
// 
//  lineRate2: Contain the line rate(for channel 4 to 7)
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
// * Configuration will be lost after applying Linerate
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_4054(mlbertapi* instance, double LineRate1, double LineRate2);
#pragma region declaration
//++
//
//	int mlBert_SetPRBSPattern(mlbertapi* instance, int channel, int txPattern, int rxPattern, int txInvert, int rxInvert)
//
// Routine Description:
//
//	Set the PRBS Pattern by channel along the the TX and RX Invert status
//
// Arguments:
//
//  instance :API instance
//  channel: Channel selected
//  txPattern: Pattern applied, PN7 = 0, PN9 = 1, PN15 = 2, PN23 = 3, PN31 = 4
//                            , User pattern= 5
//  rxPattern: Pattern applied, PN7 = 0, PN9 = 1, PN15 = 2, PN23 = 3, PN31 = 4
//                             , Auto Detect Pattern = 5
//  txInvert: Set the TX Invert Status Normal Mode = 0, Inverted Mode = 1
//  rxInvert: Set the RX Invert Status Normal Mode = 0, Inverted Mode = 1
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetPRBSPattern(mlbertapi* instance, int channel, int txPattern, int rxPattern, int txInvert, int rxInvert);
#pragma region declaration
//++
//
//	int mlBert_SetTxUserPattern(mlbertapi* instance, int channel, unsigned long long UserDefinedPattern)
//
// Routine Description:
//
//	Set User Defined patter for ML4009, ml4009 jit ml4009eql, ml4039, ml 4039jit
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  UserDefinedPattern: User Defined Pattern Applied in Dec
//                      NB: If Linerate > 21 pattern is 40 bit length
//                          If Linerate <= 15 pattern is 16 bit length
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetTxUserPattern(mlbertapi* instance, int channel, unsigned long long UserDefinedPattern);
#pragma region declaration
//++
//
//	int mlBert_SetTxUserPattern39B(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternMSB, unsigned long long UserDefinedPatternLSB)
//
// Routine Description:
//
//	Set User Defined patter for ML4039b
//
// Arguments:
//
//  channel: Channel selected
//  UserDefinedPatternMSB: User Defined Pattern Applied in Dec MSB
//  UserDefinedPatternLSB: User Defined Pattern Applied in Dec LSB
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetTxUserPattern39B(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternMSB, unsigned long long UserDefinedPatternLSB);
#pragma region declaration

//
//	int mlBert_SetTxUserPatternPAM(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternLo, unsigned long long UserDefinedPatternMid, unsigned long long UserDefinedPatternHi)
//
// Routine Description:
//
//	Set User Defined patter for PAM
//
// Arguments:
//
//  channel: Channel selected
//  UserDefinedPatternLo
//  UserDefinedPatternMid
//  UserDefinedPatternHi
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetTxUserPatternPAM(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternLo, unsigned long long UserDefinedPatternMid, unsigned long long UserDefinedPatternHi);

#pragma region declaration

//
//	int mlBert_SetTxUserPatternVega(mlbertapi* instance, int channel, unsigned long UserDefinedPatternLo, unsigned long UserDefinedPatternMid)
//
// Routine Description:
//
//	Set User Defined patter for 4039B
//
// Arguments:
//
//  channel: Channel selected
//  UserDefinedPatternLo
//  UserDefinedPatternMid
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetTxUserPatternVega(mlbertapi* instance, int channel, unsigned long UserDefinedPatternLo, unsigned long UserDefinedPatternMid);

#pragma region declaration

//
//	int mlBert_SetTxUserPatternPolaris(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternLo, unsigned long long UserDefinedPatternMid, UINT16 repetition1, UINT16 repetition2)
//
// Routine Description:
//
//	Set User Defined patter for 4039D, 4079D, 4039E
//
// Arguments:
//
//  channel: Channel selected
//  UserDefinedPatternLo
//  UserDefinedPatternMid
//	repetition1
//	repetition2
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetTxUserPatternPolaris(mlbertapi* instance, int channel, unsigned long long UserDefinedPatternLo, unsigned long long UserDefinedPatternMid, UINT16 repetition1, UINT16 repetition2);

#pragma region declaration
//++
//
//	int mlBert_CML_OutputDriver(mlbertapi* instance, int channel, double value)
//
// Routine Description:
//
//	Change the output level of the signal in ML4004
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: output level of the signal in mV, between 0-3
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_CML_OutputDriver(mlbertapi* instance, int channel, double value);
#pragma region declaration
//++
//
//	int mlBert_OutputLevel(mlbertapi* instance, int channel, double value)
//
// Routine Description:
//
//	Change the output level of the signal
//
// Arguments:
//
//  instance :API instance
//  channel: Channel selected
//  Value: output level of the signal in mV, between 20-1200
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_OutputLevel(mlbertapi* instance, int channel, double value);
#pragma region declaration
//++
//
//	int mlBert_ML403bHVAmpltiude(mlbertapi* instance, double AmpltiudeValue, double RiseTimeAmpltiude, double FallTimeAmpltiude)
//
// Routine Description:
//
//	Change the output level of the signal in HV channel Ml403b
//
// Arguments:
//
//    instance :API instance
//  AmpltiudeValue: output level of the signal in steps
//  RiseTimeAmpltiude: output level of the RiseTime
//  FallTimeAmpltiude: output level of the FallTime
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ML403bHVAmpltiude(mlbertapi* instance, double AmpltiudeValue, double RiseTimeAmpltiude, double FallTimeAmpltiude);

#pragma region declaration
//++
//
//	int mlBert_InnerEyeLevel(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Change the output level of the Inner Eye in Ml4004
//
// Arguments:
//
//  channel: Channel selected
//  value  : Amplitude for the inner eye, this value is form 0 to 15
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_InnerEyeLevel(mlbertapi* instance, int channel, int value);
#pragma region declaration
//++
//
//	int mlBert_ErrorInserationMode(mlbertapi* instance, int channel, UInt16 errorCountToInject)
//
// Routine Description:
//
//	Insert Specific count ofErrors in a signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  errorCountToInject:  Number of error that need to be inserted
//                      NB: If Linerate > 21 pattern is 40 at a time
//                          If Linerate <= 15 pattern is 32 at a time
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ErrorInserationMode(mlbertapi* instance, int channel, UInt16 errorCountToInject);

////////////////////////////////////////////////////////////////////////////////////////////
// Board Monitor
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_Monitor(mlbertapi* instance, byte Channel, byte StatusID, double *Data)
//
// Routine Description:
//
//	Change the output level of the signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  StatusID: Status ID to monitor followed by the Enumeration below
//                M_ClockIn, M_LineRate, M_ClockOut, M_TX_enable, M_RX_enable,
//                M_TXpattern, M_CustomPattern, M_TXPattern_length, M_TXinvert, M_OutputLevel,
//                M_RXpattern, M_RXinvert, M_RXPatternLength, M_PatternLock,
//                M_PreEmphasis, M_PostEmphasis,
//                M_ErrorIns, M_RXDFEValue,
//                M_BERTimer, M_BERPhase, M_BERVerticalOffset,
//                M_PhaseSkew, M_PMFreq, M_PMAmp, M_PMRJ, M_AMRJ
//  Data : Value returned
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Monitor(mlbertapi* instance, byte Channel, byte StatusID, double *Data);
#pragma region declaration
//++
//
//	double mlBert_TempRead(mlbertapi* instance, int index)
//
// Routine Description:
//
//	Return the temperature of the board
//
// Arguments:
//
//    instance :API instance
//  index: Two temp sense are on the board index 0 and 1
//
// Return Value:
//
//    Returns the temperature in degree Celsius
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_TempRead(mlbertapi* instance, int index);
#pragma region declaration
//++
//
//	int mlBert_PRBSVerfication(mlbertapi* instance, int channel)
//
// Routine Description:
//
//	Change the output level of the signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_PRBSVerfication(mlbertapi* instance, int channel);

////////////////////////////////////////////////////////////////////////////////////////////
// BERT
////////////////////////////////////////////////////////////////////////////////////////////
#pragma region declaration
//++
//
//	int mlBert_SetBERCounter(mlbertapi* instance, UInt32 packetCounter)
//
// Routine Description:
//
//	Set the BER Timer in ms
//
// Arguments:
//
//    instance :API instance
//  counterIn25KBits: set packet count (packet = 25K Bit)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_SetBERCounter(mlbertapi* instance, UInt32 packetCounter);

//++
//
//	int mlBert_ChangeBERPhase(mlbertapi* instance, int channel, byte Phase, byte VerticalOffset)
//
// Routine Description:
//
//	Set Phase and Vertical offset for Instant BER
//  It allows to specify any location on the eyeContour and apply a BER Test
//
// Arguments:
//
//    instance :API instance
//   Phase: phase value (0-128)
//   VerticalOffset: Vertical Offset (0-255)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_ChangeBERPhase(mlbertapi* instance, int channel, byte Phase, byte VerticalOffset);

//++
//
// int mlBert_ChangeBERPhaseAndOffset_pS_mV(mlbertapi* instance, int Channel, int phase, int Amplitude)
//
// Routine Description:
//
// to change the phase and amplitude desicion point
//
// Arguments:
//    instance  : API instance
//    Channel   : channel number
//    Phase     : phase value value in ps
//    Amplitude : Amplitude level in mv
// Return Value:
//
//    Returns the function status
//
//--
APICALL int  STACKMODE mlBert_ChangeBERPhaseAndOffset_pS_mV(mlbertapi* instance, int Channel, int phase, int Amplitude);

//++
//
//	int mlBert_DoInstantBER(mlbertapi* instance, double berValues[])
//
// Routine Description:
//
//	Run BER Test and return the BER for the specified BitCount in SetBERCounter function
//
// Arguments:
//
//    instance :API instance
//  berValues: Return BER Value in a table for all 4 channels, need to send double table[4]
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_DoInstantBER(mlbertapi* instance, double berValues[]);

//++
//
//	int mlBert_RealTimeBEREnable(mlbertapi* instance, int enable)
//
// Routine Description:
//
//	Enable or Disable internal board BER
//
// Arguments:
//
//    instance :API instance
//  enable: Enable = 1 or Disable = 0 Bathtub Saving.
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_RealTimeBEREnable(mlbertapi* instance, int enable);

//++
//
//	int mlBert_ReadErrorCorrectionPolaris(mlbertapi* instance, int channelIdx, UInt64 errorCorrected[], UInt64 BlockCount[], UInt64 SaturatedBlockCount[])
//
// Routine Description:
//
//	Read Error Correction for 4039D
//
// Arguments:
//
//    instance :API instance
//  channelIdx
//	errorCorrected: corrected errors obtained
//	BlockCount
//	SaturatedBlockCount
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_ReadErrorCorrectionPolaris(mlbertapi* instance, int channelIdx, UInt64 errorCorrected[], UInt64 BlockCount[], UInt64 SaturatedBlockCount[]);
//++
//
//	int mlBert_ReadRealTimeBER(mlbertapi* instance, UInt64 errorCount[], UInt32 * TimeSpendIns)
//
// Routine Description:
//
//	Enable or Disable internal board BER
//
// Arguments:
//
//    instance :API instance
//  errorCount: Return Bit Error Count in a table for all 4 channels, need to send double UInt64[4]
//  TimeSpendIns : Return the Accumulated time
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_ReadRealTimeBER(mlbertapi* instance, UInt64 errorCount[], UInt32 * TimeSpendIns);

//++
//
//	int mlBert_BathTubVerticalOffset(mlbertapi* instance, int channel, byte VerticalOffset)
//
// Routine Description:
//
//	Set Vertical offset for Instant Bathtub
//
// Arguments:
//
//    instance :API instance
//   VerticalOffset: Vertical Offset (0-255)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_BathTubVerticalOffset(mlbertapi* instance, int channel, byte VerticalOffset);

//++
//
//	int mlBert_BathTubVerticalOffset_mv(mlbertapi* instance, int channel, byte VerticalOffset)
//
// Routine Description:
//
//	Set Vertical offset for Instant Bathtub
//
// Arguments:
//   instance : API instance
//   channel  : Channel number   
//   VerticalOffset: Vertical Offset in mv
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_BathTubVerticalOffset_mv(mlbertapi* instance, int channel, byte VerticalOffset);

//++
//
// int mlBert_GetBathTub(mlbertapi* instance, int channel, double *xValues, double *BERValues)
//
// Routine Description:
//
// this function is used to read the Bathtub for a specific channel
//
// Arguments:
//
// instance :API instance
// channel: the Channel number
// xValues: return the XValues in an array of 128 values
// BERValues: return the BERValues of a specific xValue in an array of 128 values
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetBathTub(mlbertapi* instance, int channel, double *xValues, double *BERValues);

//++
//
// int mlBert_GetPam4Histogram(mlbertapi* instance, int channel, double *xValues, double *yValues)
//
// Routine Description:
//
// this function is used to read the Pam4 Histogram for a specific channel
//
// Arguments:
//
// instance :API instance
// channel: the Channel number
// xValues: return the XValues in an array of 160 values
// yValues: return the YValues of a specific xValue in an array of 160 values
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetPam4Histogram(mlbertapi* instance, int channel, double *xValues, double *yValues);

//++
//
// int mlBert_GetPam4SignalToNoiseRatio(mlbertapi* instance, int channel, double *Values)
//
// Routine Description:
//
// this function is used to read the Pam4 Signal To Noise Ratio for a specific channel
//
// Arguments:
//
// instance :API instance
// channel: the Channel number
// Values: return the ratio 
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetPam4SignalToNoiseRatio(mlbertapi* instance, int channel, double *Values);

//++
//
// int mlBert_GetBathTubparallel(mlbertapi* instance, int *channel, double *xValues, double *BERValues)
//
// Routine Description:
//
// this function is used to read the Bathtub for a all the channels, it's only used for the ML4039-eql
//
// Arguments:
//  instance :API instance
// channel: the Channels configuration
// xValues: return the XValues for each bathtub in an array of 64 values each
// BERValues: return the BERValues for each bathtub in an array of 64 values each
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetBathTubparallel(mlbertapi* instance, int *channel, double *xValues, double *BERValues);

//++
//
// int mlBert_GetVBathTub(mlbertapi* instance, int channel, byte HorizontalOffest, double *xValues, double *BERValues)
//
// Routine Description:
//
// this function is used to read the Bathtub for a specific channel
//
// Arguments:
//
// instance :API instance
// channel: the Channel number
// HorizontalOffest
// xValues: return the XValues in an array of 128 values
// BERValues: return the BERValues of a specific xValue in an array of 128 values
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetVBathTub(mlbertapi* instance, int channel, byte HorizontalOffest, double *xValues, double *BERValues);

APICALL int  STACKMODE mlBert_GetFullSweepBathTub(mlbertapi* instance, int channel, unsigned long long packetWaitCount, byte start, byte Stop, double *xValues, double *BERValues);
//++
//
// int mlBert_CalculateBathtubDualDirac(mlbertapi* instance, int channel, double *xValues, double *BERValues)
//
// Routine Description:
//
// This function apply Dual Dirac on the pre-captured bathtub
// ND: Run APIGetBathtub before this function is required
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// xValues: return the XValues in an array of 128 values
// BERValues: return the BERValues of a specific xValue in an array of 128 values
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_CalculateBathtubDualDirac(mlbertapi* instance, int channel, double *xValues, double *BERValues);

//++
//
// int mlBert_CalculateVerticalBathtubDualDirac(mlbertapi* instance, int channel, double *xValues, double *BERValues)
//
// Routine Description:
//
// This function apply Dual Dirac on the pre-captured vertical bathtub
// ND: Run APIGetBathtub before this function is required
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// xValues: return the XValues in an array of 255 values
// BERValues: return the BERValues of a specific xValue in an array of 255 values
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_CalculateVerticalBathtubDualDirac(mlbertapi* instance, int channel, double *xValues, double *BERValues);

//++
//
// bool mlBert_ReadRJDJ(mlbertapi* instance, int channel, double *DJ_array, double *RJ_array)
//
// Routine Description:
//
// this function is used to read the DJ and RJ
//
// Arguments:
//
// instance :API instance
// DJ_array: DJ array, for the 4 channels
// RJ_array: RJ array, for the 4 channels
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_ReadRJDJ(mlbertapi* instance, int channel, double *DJ_array, double *RJ_array);

//++
//
// bool mlBert_ReadVerticalRJDJ(mlbertapi* instance, int channel, double *DJ_array, double *RJ_array)
//
// Routine Description:
//
// this function is used to read the DJ and RJ for a vertical bathtub
//
// Arguments:
//
// instance :API instance
// DJ_array: DJ array, for the 4 channels
// RJ_array: RJ array, for the 4 channels
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_ReadVerticalRJDJ(mlbertapi* instance, int channel, double *DJ_array, double *RJ_array);
//++
//
// int mlBert_CalculateJitter(mlbertapi* instance, int channel, double targetBER, double* jitterMeasurements)
//
// Routine Description:
//
// this function is used to read the Bathtub for a specific channel
//
// Arguments:
//
//    instance :API instance
// channel: The Channel number
// targetBER: Measure any BER ex: J2 = 10^-2, J9 = 10^-9
// jitterMeasurements: return the jitter measurements on that specific targetBER
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_CalculateJitter(mlbertapi* instance, int channel, double targetBER, double* jitterMeasurements);

//++
//
// int mlBert_CalculateVerticalJitter(mlbertapi* instance, int channel, double targetBER, double* jitterMeasurements)
//
// Routine Description:
//
// this function is used to read the vertical Bathtub for a specific channel
//
// Arguments:
//
//    instance :API instance
// channel: The Channel number
// targetBER: Measure any BER ex: J2 = 10^-2, J9 = 10^-9
// jitterMeasurements: return the jitter measurements on that specific targetBER
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_CalculateVerticalJitter(mlbertapi* instance, int channel, double targetBER, double* jitterMeasurements);

//++
//
// int mlBert_GetEye(mlbertapi* instance, int channel, double *xValues, double *yValues, double *berValues)
//
// Routine Description:
//
// this function is used to read the Eye for a specific channel
// It returns the matrix with xValues, yValues and berValues for every specific Location
// The array should be 32768 stands for 128 phase step and 256 vertical offset
//
// Arguments:
//
//    instance :API instance
// Channel: the Channel number
// xValues: X location in the matrix
// yValues: Y location in the matrix
// berValues: BER Values on this specific X and Y Locations
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetEye(mlbertapi* instance, int channel, double *xValues, double *yValues, double *berValues);

//++
//
// int mlBert_GetEyeparallel(mlbertapi* instance, int *channel, int PacketNumber, double *xValues, double *yValues, double *berValues)
//
// Routine Description:
//
// this function is used to read the Eye for a set if channels in parallel mode, it's only used for ML4039B-eql
// It returns the matrix with xValues, yValues and berValues for every specific Location
// The array should be 32768 stands for 128 phase step and 256 vertical offset
//
// Arguments:
//
//    instance :API instance
// Channel: the Channel number
// xValues: X location in the matrix
// yValues: Y location in the matrix
// berValues: BER Values on this specific X and Y Locations
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetEyeparallel(mlbertapi* instance, int *channel, int PacketNumber, double *xValues, double *yValues, double *berValues);

//++
//
// int mlBert_GetEyeMeasurements(mlbertapi* instance, int channel, double targetBER,
//double* eyeWidthStart, double* eyeWidthEnd,
//double* eyeHeightStart, double* eyeHeightEnd)
//
// Routine Description:
//
// It Measures the eye width and eye height for a specific BER.
// NB: It required to do GetEye first
//
// Arguments:
//
//    instance :API instance
// Channel: the Channel number
// targetBER: measure any BER ex: 10^-2, 10^-4
// eyeWidthStart: return Eye width start X location for the specified targetBER
// eyeWidthEnd: return Eye width end X location for the specified targetBER
// eyeHeightStart: return Eye height start Y location for the specified targetBER
// eyeHeightEnd: return Eye height end Y location for the specified targetBER
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_GetEyeMeasurements(mlbertapi* instance, int channel, double targetBER,
	double* eyeWidthStart, double* eyeWidthEnd,
	double* eyeHeightStart, double* eyeHeightEnd);
//++
//
// int mlBert_FastDiamond(mlbertapi* instance, int Channel, double * EyeWidth, double * EyeHeight)
//
// Routine Description:
//
// It Measures the eye width and eye height for a specific BER.
// NB: It required to do GetEye first
//
// Arguments:
//
//    instance :API instance
// Channel: the Channel number
// EyeWidth: return Eye Width
// EyeHeight: return Eye Height
//
// Return Value:
//
//    Returns if the command was executed properly Success = true, failed = false
//
//--
APICALL int  STACKMODE mlBert_FastDiamond(mlbertapi* instance, int Channel, double * EyeWidth, double * EyeHeight);

//++
//
// int mlBert_CalculateMaskMeasurements(mlbertapi* instance, int channel,
//double x1, double x2, double x3, double x4, double x5, double x6,
//double y0, double y1, double y2, double y3, double y4, double y5, double y6)
//
// Routine Description:
//
// It counts the number of failing points
// NB: It required to do GetEye first
//
// Arguments:
//
//    instance :API instance
//    Channel: the Channel number
//    x1-x6: Mask X Values in ps
//	  y0-y6: Mask Y Values in mV (NB: since we are using eye mask in eye contour Set Y3 to -1 and Y4 to 500)
//
// Return Value:
//
//    Returns number of failing points
//
//--
APICALL int  STACKMODE mlBert_CalculateMaskMeasurements(mlbertapi* instance, int channel,
	double x1, double x2, double x3, double x4, double x5, double x6,
	double y0, double y1, double y2, double y3, double y4, double y5, double y6);

////////////////////////////////////////////////////////////////////////////////////////////
// Emphasis Control Works only on Non-Jits versions
////////////////////////////////////////////////////////////////////////////////////////////
//++
//
//	int mlBert_PreEmphasis(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the PreEmphasis settings on the signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PreEmphasis
//			  -ML4039 between 0-100
//            -ML4039-Jit Not Applied
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_PreEmphasis(mlbertapi* instance, int channel, int value);

//++
//
//	int mlBert_PostEmphasis(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the PostEmphasis settings on the signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039 between 0-100
//            -ML4039-Jit Not Applied
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_PostEmphasis(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap0(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap0 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap0(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap1(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap1 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap1(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap2(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap2 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap2(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap3(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap3 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap3(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap4(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap4 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap4(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap5(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap5 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap5(mlbertapi* instance, int channel, int value);
//++
//
//	int mlBert_Tap6(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//	Apply the tap6 of the 7 taps 
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  Value: The value of the PostEmphasis
//			  -ML4039D/79D/39B/39E: between -1000 and 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_Tap6(mlbertapi* instance, int channel, int value);

//++
//
// int mlBert_DFEEnable(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
// Enable Disable the DFE Settings
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// status: true or false.
//			  -ML4039 between 0-100
//            -ML4039-Jit Not Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_DFEEnable(mlbertapi* instance, int channel, int status);

//++
//
// int mlBert_DFESetValue(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
// Change the DFE Settings
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// value: selection index is following this enumeration
//        0.67, 1.34, 2.01, 2.68, 3.35, 4.02, 4.69, 5.36,
//        6.03, 6.7, 7.37, 8.04, 8.71, 9.38, 10
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_DFESetValue(mlbertapi* instance, int channel, int value);

//++
//
// int mlBert_RXEnable(mlbertapi* instance, int channel, bool Status)
//
// Routine Description:
//
// enable Disable the RX Line
//
// Arguments:
// instance :API instance
// channel = indicate the channel number.
// Status = indicate the RX Line status
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_RXEnable(mlbertapi* instance, int channel, bool Status);

//++
//
// int mlBert_TXEnable(mlbertapi* instance, int channel, bool Status)
//
// Routine Description:
//
// enable Disable the RX Line
//
// Arguments:
// instance :API instance
// channel = indicate the channel number.
// Status = indicate the TX Line status
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_TXEnable(mlbertapi* instance, int channel, bool Status);

////////////////////////////////////////////////////////////////////////////////////////////
// Signal Modulation works only on Jits Versions
////////////////////////////////////////////////////////////////////////////////////////////

//++
//
//	int mlBert_PhaseSkew(mlbertapi* instance, int channel, double amplitude)
//
// Routine Description:
//
//	Change the TX Phase Skew of the signal
//
// Arguments:
//
//    instance :API instance
//  channel: Channel selected
//  amplitude: TX Phase Skew value of the signal, between 0-4096
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_PhaseSkew(mlbertapi* instance, int channel, double amplitude);

//++
//
// int mlBert_PMEnable(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
// This function is for enabling and disabling the PM
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// status : 0 = disable, 1 = enable.
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_PMEnable(mlbertapi* instance, int channel, int status);

//++
//
// int mlBert_EnableSJRJBUJ(mlbertapi* instance, int Channel, int Status)
//
// Routine Description:
//
// This function is for enabling and disabling the SJ RJ BUJ
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// status : 0 = disable, 1 = enable.
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_EnableSJRJBUJ(mlbertapi* instance, int Channel, int Status);

//++
//
// int mlBert_SetPMFrequency(mlbertapi* instance, int channel, double frequency)
//
// Routine Description:
//
// this function set the PM Frequency
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// frequency: the value of the frequency in MHz, between 0 and 100 MHz
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
APICALL int  STACKMODE mlBert_SetPMFrequency(mlbertapi* instance, int channel, double frequency);
#pragma region declaration 
//++
//
// int mlBert_PMSJAmplitude(mlbertapi* instance, int channel, double value)
//
// Routine Description:
//
// this function change the SJ Value.
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// value: the value of the SJ, it's between 0 and 4095
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_PMSJAmplitude(mlbertapi* instance, int channel, double value);

#pragma region declaration 
//++
//
// int mlBert_PMRJEnable(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
// this function is for enabling and disabling the RJ
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// status : 0 = disable, 1 = enable.
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_PMRJEnable(mlbertapi* instance, int channel, int status);
#pragma region declaration 
//++
//
// int mlBert_PMRJAmplitude(mlbertapi* instance, int channel, double value)
//
// Routine Description:
//
// this function change the Diff Noise Value.
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// value: the value of the Diff Noise, it's between 0 and 4095
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_PMRJAmplitude(mlbertapi* instance, int channel, double value);
#pragma region declaration 
//++
//
// int mlBert_AMEnable(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
// this function is for enabling and disabling the AM
//
// Arguments:
//
//    instance :API instance
// channel: the Channel number
// status : 0 = disable, 1 = enable.
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--

#pragma endregion
APICALL int  STACKMODE mlBert_AMEnable(mlbertapi* instance, int channel, int status);
#pragma region declaration 
//++
//
// int mlBert_AMRJAmplitude(mlbertapi* instance, int channel, double value)
//
// Routine Description:
//
// this function change the RJ Value.
//
// Arguments:
//
// instance :API instance
// channel: the Channel number
// value: the value of the RJ, it's between 0 and 4095
//			-ML4039 Not Applied
//          -ML4039-Jit Applied
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_AMRJAmplitude(mlbertapi* instance, int channel, double value);

#pragma region declaration 
//++
//
// int mlBert_UnOptimizedSJ(mlbertapi* instance, int Channel)
//
// Routine Description:
//
// To disable the Sj calibration
//
// Arguments:
//
//    instance :API instance
//    channel: channel index
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_UnOptimizedSJ(mlbertapi* instance, int Channel);
#pragma region declaration 
//++
//
// int mlBert_UnOptimizedPMRJ(mlbertapi* instance, int Channel)
//
// Routine Description:
//
// To disable the PM RJ calibration
//
// Arguments:
//
//    instance :API instance
//    channel: channel index
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--

#pragma endregion
APICALL int  STACKMODE mlBert_UnOptimizedPMRJ(mlbertapi* instance, int Channel);

#pragma region declaration 
//++
//
// int mlBert_LoadDefaultCalibration(mlbertapi* instance)
//
// Routine Description:
//
// Just for Testing in case the board is not calibrated, It will load default calibration
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LoadDefaultCalibration(mlbertapi* instance);

#pragma region declaration 
//++
//
// int mlBert_LoadCalibrationValue(mlbertapi* instance)
//
// Routine Description:
//
// Load calibration values
//
// Arguments:
//
//    instance :API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LoadCalibrationValue(mlbertapi* instance);

#pragma region declaration 
//++
//
// int mlBert_OptimizedSJ(mlbertapi* instance, int Channel, double* MinMax)
//
// Routine Description:
//
// To enable the SJ optimisation for a sepcifique channel
//
// Arguments:
//
//    instance :API instance
//    Channel : Channel index
//    MinMax  : array that containe the min and max for each SJ frequency { 5 Mhz, 10 Mhz, 20 Mhz, 40 Mhz, 80 Mhz };
// Return Value:
//
//    Returns the currnet calibration number, 0 if it's not calibrated
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_OptimizedSJ(mlbertapi* instance, int Channel, double* MinMax);

#pragma region declaration 
//++
//
// int  mlBert_GetSkewOptimizedValue(mlbertapi* instance, int Channel)
//
// Routine Description:
//
// To get skew optimized value
//
// Arguments:
//
//    instance :API instance
//    Channel : Channel index
// Return Value:
//
//    Returns the currnet calibration number, 0 if it's not calibrated
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_GetSkewOptimizedValue(mlbertapi* instance, int Channel);
#pragma region declaration 
//++
//
// int mlBert_OptimizedPMRJ(mlbertapi* instance, int Channel, double* MinMax)
//
// Routine Description:
//
// To enable the PM RJ optimisation for a sepcifique channel
//
// Arguments:
//
//    instance :API instance
//    Channel : Channel index
//    MinMax  : array that containe the min and max for each SJ frequency { 5 Mhz, 10 Mhz, 20 Mhz, 40 Mhz, 80 Mhz };
// Return Value:
//
//    Returns the currnet calibration number, 0 if it's not calibrated
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_OptimizedPMRJ(mlbertapi* instance, int Channel, double* MinMax);
#pragma region declaration 
//++
//
// int mlBert_ReadCalibrationValues(mlbertapi* instance, double* values)
//
// Routine Description:
//
// to read the calibration values in the EEPROM
//
// Arguments:
//
//    instance :API instance
//    values : array of double that read the calibration value
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadCalibrationValues(mlbertapi* instance, double* values);
#pragma region declaration 
//++
//
// int mlBert_LoadDynamicStressers(mlbertapi* instance, int *Data)
//
// Routine Description:
//
// to set the Dynamic stresses configuration
//
// Arguments:
//
//    instance :API instance
//    Data : array of double the stressers data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LoadDynamicStressers(mlbertapi* instance, int *Data);
#pragma region declaration 
//++
//
// int mlBert_EnableDynamicStressers(mlbertapi* instance, int channel, int timer, int Interpolator_Step, int enable)
//
// Routine Description:
//
// to set the Dynamic stresses configuration
//
// Arguments:
//
//    instance :API instance
//    timer : time between each stresser (in ms)
//    enable : 1 to enable it and 0 to disable it
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_EnableDynamicStressers(mlbertapi* instance, int channel, int timer, int Interpolator_Step, int enable);

#pragma region declaration 
//++
//
// int mlBert_Check3BRevision(mlbertapi* instance, int *Revision, int *SFPRevision, int *HVRevision, int *XFPRevision)
//
// Routine Description:
//
// to check the Ml4003b revision number
//
// Arguments:
//
//    instance :API instance
//    LineRateRevision : contain the Revision number, 1 Bert Only (below 16 G) 2 30 G
//    SFPRevision      : 0 NO SFP 1 SFP 
//    HVRevision       : 0 NO HD 1 HD
//    XFPRevision       : 0 NO XFP 1 XFP
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_Check3BRevision(mlbertapi* instance, int *Revision, int *SFPRevision, int *HVRevision, int *XFPRevision);
#pragma region declaration 
//++
//
// int mlBert_CheckEQLRevision(mlbertapi* instance, int *Revision)
//
// Routine Description:
//
// to check the Ml4009-EQL revision number
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, 0xx for rev a 0xb for rev b
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_CheckEQLRevision(mlbertapi* instance, int *Revision);
#pragma region declaration 
//++
//
// int mlBert_Check39B_EQLHVRevision(mlbertapi* instance, int *Revision)
//
// Routine Description:
//
// to check the Ml4039B-EQL board Type
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, 0- for no HV and 1 for HV version
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check39B_EQLHVRevision(mlbertapi* instance, int *Revision);
#pragma region declaration 
//++
//
// int mlBert_Check39A_CleanJitRevision(mlbertapi* instance, int *Revision)
//
// Routine Description:
//
// to check the Ml4039A revision number
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, 0- for clean , 1 for Jit version, 2 for clean and jit  
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check39A_CleanJitRevision(mlbertapi* instance, int *Revision);
#pragma region declaration 
//++
//
// int mlBert_Check39Brv2_CleanJitRevision(mlbertapi* instance, int *Revision)
//
// Routine Description:
//
// to check the Ml4039Brv2 revision type
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, 0- for clean , 1 for Jit version
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check39Brv2_CleanJitRevision(mlbertapi* instance, int *Revision);
#pragma region declaration 
//++
//
// int mlBert_CheckTDA_Revision(mlbertapi* instance, int *Revision, int *BoardRV, int *IsPortable, int *Is407)
//
// Routine Description:
//
// to check the Ml4009-EQL revision number
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, 0- for no HV or jit, 1 for HV version, 2 for jit revsion and 3 for jit and HV revsion
//    BoardRV  : contain the Board Revision 1 RevB and 0 Rev A
//    IsPortable : if true taht means this is a Portable version
//    Is407: if 1 contains 407 circuit, if 0 doesn't contain 407 circuit
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_CheckTDA_Revision(mlbertapi* instance, int *Revision, int *BoardRV, int *IsPortable, int *Is407);

#pragma region declaration 
//++
//
// int mlBert_Check4039Jit_Revision(mlbertapi* instance, int *Revision, int *WithBUJ)
//
// Routine Description:
//
// to check the Ml4039 jit revision number, and Buj option
//
// Arguments:
//
//    instance :API instance
//    Revision : contain the Revision number, c = Rev C else old rev
//    WithBUJ  : if 1 BUJ exist if 0 BUJ don't exist
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check4039Jit_Revision(mlbertapi* instance, int *Revision, int *WithBUJ);

#pragma region declaration
//++
//
// int mlBert_Check4039EVBRev(mlbertapi* instance, int *EVBRevision)
//
// Routine Description:
//
// to check the Ml4039 EVB revision number
//
// Arguments:
//
//    instance :API instance
//    EVBRevision : contain the Revision number, 1 = Rev B else old rev
// Return Value:
//
//    Returns if the board is a EVB or not
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check4039EVBRev(mlbertapi* instance, int *EVBRevision);
#pragma region declaration
//++
//
// int mlBert_Check4039PolariseRev(mlbertapi* instance, int *PolariseRevision)
//
// Routine Description:
//
// to check the Ml4039D revision number
//
// Arguments:
//
//    instance :API instance
//    PolariseRevision : 1 = 4079D, 0 = 4039D
// Return Value:
//
//   Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_Check4039PolariseRev(mlbertapi* instance, int *PolariseRevision);

#pragma region declaration 
//++
//
// int  mlBert_EnableInternalLoopbcak(mlbertapi* instance, int enable)
//
// Routine Description:
//
// to enable the internal loopback
//
// Arguments:
//
//    instance :API instance
//    enable : 1 to enable, 0 to disable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_EnableInternalLoopbcak(mlbertapi* instance, int enable);
#pragma region declaration 
//++
//
// int mlBert_WriteEqualizerRegister(mlbertapi* instance, int Channel, UInt16 Register, UInt16 Data)
//
// Routine Description:
//
// to write into a eql tab
//
// Arguments:
//
//    instance : API instance
//    Channel  : channel number
//    Register : Register Value
//    Data     : data to be applied
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_WriteEqualizerRegister(mlbertapi* instance, int Channel, UInt16 Register, UInt16 Data);
#pragma region declaration 
//++
//
// int mlBert_ReadEqualizerRegister(mlbertapi* instance, int Channel, UInt16 Register, UInt16 *Data)
//
// Routine Description:
//
// to read from a eql tab
//
// Arguments:
//
//    instance : API instance
//    Channel  : channel number
//    Register : Register Value
//    Data     : tab value extracted
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_ReadEqualizerRegister(mlbertapi* instance, int Channel, UInt16 Register, UInt16 *Data);
#pragma region declaration 
//++
//
// int mlBert_SkewCalibration(mlbertapi* instance, int Channel, int mode, double Steps_UI)
//
// Routine Description:
//
// to converte skew steps to UI 
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    mode      : 0 to convert steps to UI , 1 to convert UI to steps
//    Steps_UI    : step or UI value
// Return Value:
//
//    Returns the equililant steps in UI
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_SkewCalibration(mlbertapi* instance, int Channel, int mode, double Steps_UI);
#pragma region declaration 
//++
//
// int mlBert_ML4039BEQLAmplitudeCalibration_Step_T0_MV(mlbertapi* instance, int Channel, int Step, double LineRate)
//
// Routine Description:
//
// to converte skew steps to UI 
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    Step     : step value 
//    LineRate : line Rate value for this channel
// Return Value:
//
//    Returns the equililant steps in MV
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_ML4039BEQLAmplitudeCalibration_Step_T0_MV(mlbertapi* instance, int Channel, int Step, double LineRate);
#pragma region declaration 
//++
//
// int mlBert_ML4039BEQLAmplitudeCalibration_MV_T0_Step(mlbertapi* instance, int Channel, int _MV, double LineRate)
//
// Routine Description:
//
// to converte skew steps to UI 
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    _MV       : MV value 
//    LineRate : line Rate value for this channel
// Return Value:
//
//    Returns the equililant MV in step
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_ML4039BEQLAmplitudeCalibration_MV_T0_Step(mlbertapi* instance, int Channel, int _MV, double LineRate);

#pragma region declaration 
//++
//
// int mlBert_ML4039BEQLAmplitudeHVCalibration_Step_T0_MV(mlbertapi* instance, int Channel, int Step, double LineRate)
//
// Routine Description:
//
// to converte skew steps to UI 
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    Step     : step value 
//    LineRate : line Rate value for this channel
// Return Value:
//
//    Returns the equililant steps in MV
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_ML4039BEQLAmplitudeHVCalibration_Step_T0_MV(mlbertapi* instance, int Channel, int Step, double LineRate);
#pragma region declaration 
//++
//
// int mlBert_ML4039BEQLAmplitudeHVCalibration_MV_T0_Step(mlbertapi* instance, int Channel, int _MV, double LineRate)
//
// Routine Description:
//
// to converte skew steps to UI 
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    _MV       : MV value 
//    LineRate : line Rate value for this channel
// Return Value:
//
//    Returns the equililant MV in step
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_ML4039BEQLAmplitudeHVCalibration_MV_T0_Step(mlbertapi* instance, int Channel, int _MV, double LineRate);
#pragma region declaration 
//++
//
// int mlBert_WriteStatusToEEProm(mlbertapi* instance, int Channel, UInt32 Address, double Data)
//
// Routine Description:
//
// to write status to EEPROM
//
// Arguments:
//
//    instance :API instance
//    Channel  : channel number
//    Address 
//    Data
// Return Value:
//
//   Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_WriteStatusToEEProm(mlbertapi* instance, int Channel, UInt32 Address, double Data);

////////////////////////////////////////////////QSFP////////////////////////////////////////////
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_SetRegister(mlbertapi* instance, int Module, UInt16 Register, UInt16 Value)
//
// Routine Description:
//
//    Set a specfic register from a QSFP Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Register : contain the register value
//    Value :  Data that need to be inserted in the register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_SetRegister(mlbertapi* instance, int Module, UInt16 Register, UInt16 Value);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_GetRegister(mlbertapi* instance, int Module, UInt16 Register, UInt16 *Value)
//
// Routine Description:
//
//    Read a specfic register from a QSFP Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Register : contain the register value
//    Value : pointer that contain the Register data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_GetRegister(mlbertapi* instance, int Module, UInt16 Register, UInt16 *Value);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_P3V3_Current_Monitor(mlbertapi* instance, int Module, double* data)
//
// Routine Description:
//
//    Read Current Senesor for Vcc of a specifce QSFP Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    data : containe the Current
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_P3V3_Current_Monitor(mlbertapi* instance, int Module, double* data);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_MODPRS(mlbertapi* instance, int Module, bool* status)
//
// Routine Description:
//
//    Check if the QSFP module is present 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    status : Pointer that containe the Module Status (true or false)
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_MODPRS(mlbertapi* instance, int Module, bool* status);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_LPMODE(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    enable diable QSFP low power mode 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means set low power mode on, false disable low power mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_LPMODE(mlbertapi* instance, int Module, bool asserted);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_RESET(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    reset QSFP Module 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means Disable QSFP, false means enable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_RESET(mlbertapi* instance, int Module, bool asserted);
#pragma region declaration 
//++
//
//	bool mlBert_QSFP_MODSEL(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    Select module for I2C communication
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means I2C communication is selectd for this module,
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_MODSEL(mlbertapi* instance, int Module, bool asserted);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_IntL(mlbertapi* instance, int Module, bool* status)
//
// Routine Description:
//
//    Check I2c Interput  
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means I2C interput is ON
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_IntL(mlbertapi* instance, int Module, bool* status);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_GetVCC(mlbertapi* instance, int Module, double* Data)
//
// Routine Description:
//
//    Read Vcc Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_GetVCC(mlbertapi* instance, int Module, double* Data);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Get_VCCTX(mlbertapi* instance, int Module, double *Data)
//
// Routine Description:
//
//    Read Vcc TX Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Get_VCCTX(mlbertapi* instance, int Module, double *Data);

#pragma region declaration
//++
//
//	bool mlBert_QSFP_Get_VCCRX(mlbertapi* instance, int Module, double *Data)
//
// Routine Description:
//
//    Read Vcc RX Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Get_VCCRX(mlbertapi* instance, int Module, double *Data);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Get_temperature(mlbertapi* instance, int Module, double *Temp)
//
// Routine Description:
//
//    Read Vcc QSFP Module Tempreature
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Temp : Contain the Tempreature value
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Get_temperature(mlbertapi* instance, int Module, double *Temp);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Get_TX_Current(mlbertapi* instance, int Module, double* Current)
//
// Routine Description:
//
//    Read QSFP TX Current Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Current : Contain the Current value
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Get_TX_Current(mlbertapi* instance, int Module, double* Current);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Get_RX_Current(mlbertapi* instance, int Module, double* Current)
//
// Routine Description:
//
//    Read QSFP RX Current Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Current : Contain the Current value
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Get_RX_Current(mlbertapi* instance, int Module, double* Current);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_SetPower(mlbertapi* instance, int Module, int Rate)
//
// Routine Description:
//
//    Set QSFP Power between 3.15V , 3.33 V and 3.45
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Rate : Contain the Voltage index 0 = 3.15, 1 = 3.33, 2 = 3.45
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_SetPower(mlbertapi* instance, int Module, int Rate);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Block_Write(mlbertapi* instance, int Module, UInt8 length, UInt8 address, byte *calib)
//
// Routine Description:
//
//    Write a block of adress into QSFP Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    length : Address length
//    address : starting address
//    calib : array of data to be writend in each address
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Block_Write(mlbertapi* instance, int Module, UInt8 length, UInt8 address, byte *calib);

#pragma region declaration 
//++
//
//	bool mlBert_QSFP_Block_Read(mlbertapi* instance, int Module, UInt8 length, UInt8 address, byte *calib)
//
// Routine Description:
//
//    Read a block of adress into QSFP Module
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    length : Address length
//    address : starting address
//    calib : array of data 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QSFP_Block_Read(mlbertapi* instance, int Module, UInt8 length, UInt8 address, byte *calib);


//////////////////////////////////////////////////////////////////SFP///////////////////////////////////////////////////
#pragma region declaration 
//++
//
//	bool mlBert_SFP_Read_SlaveAddress(mlbertapi* instance, int *SlaveAddress, int module)
//
// Routine Description:
//
//    Read SFP Slave Address
//
// Arguments:
//    Instance :API instance
//    SlaveAddress : Pointer to the Salve address
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_Read_SlaveAddress(mlbertapi* instance, int *SlaveAddress, int module);

#pragma region declaration 	
//++
//
//	bool mlBert_SFP_Set_SlaveAddress(mlbertapi* instance, int SlaveAddress, int module)
//
// Routine Description:
//
//    Set SFP Slave Address
//
// Arguments:
//    Instance :API instance
//    SlaveAddress : Slave address 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_Set_SlaveAddress(mlbertapi* instance, int SlaveAddress, int module);

#pragma region declaration 
//++
//
//	APICALL bool mlBert_SFP_Set_Register(mlbertapi* instance, int Module, UInt16 Register, USHORT SlaveAddress, UInt16 Value)
//
// Routine Description:
//
//    Set SFP Register 
//
// Arguments:
//    Instance :API instance
//    Register : Register address 
//    Value : Value for this register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_Set_Register(mlbertapi* instance, int Module, UInt16 Register, USHORT SlaveAddress, UInt16 Value);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_Get_Register(mlbertapi* instance, int module, UInt16 Register, USHORT SlaveAddress, UInt16 * Value)
//
// Routine Description:
//
//    Get SFP Register 
//
// Arguments:
//    Instance :API instance
//    Register : Register address 
//    *Value : Pointer for the Value of this register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_Get_Register(mlbertapi* instance, int module, UInt16 Register, USHORT SlaveAddress, UInt16 * Value);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_TX_FaultStatus(mlbertapi* instance, int module)
//
// Routine Description:
//
//    Check SFP RX Fault status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No TX Fault and true if TX fault 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_TX_FaultStatus(mlbertapi* instance, int module);

#pragma region declaration
//++
//
//	bool mlBert_SFP_RX_LOSStatus(mlbertapi* instance, int module)
//
// Routine Description:
//
//    Check SFP RX LOS Status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RX_LOSStatus(mlbertapi* instance, int module);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_TX_Disable(mlbertapi* instance, int Mode, int module)
//
// Routine Description:
//
//    Disable TX 
//
// Arguments:
//    Instance :API instance
//    Mode = 0 to enable TX and 1 to Disable TX
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_TX_Disable(mlbertapi* instance, int Mode, int module);


#pragma region declaration 
//++
//
//	bool mlBert_SFP_TX_DisableStatus(mlbertapi* instance, int Module)
//
// Routine Description:
//
//    Disable TX 
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if TX not disabled and true if TX Disabled
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_TX_DisableStatus(mlbertapi* instance, int Module);
#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS0_RateStatus(mlbertapi* instance, int module)
//
// Routine Description:
//
//    Select SFP RS0 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS0_RateStatus(mlbertapi* instance, int module);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS1_RateStatus(mlbertapi* instance, int module)
//
// Routine Description:
//
//    Select SFP RS1 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS1_RateStatus(mlbertapi* instance, int module);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_ReadSlaveAddress(mlbertapi* instance, int *SlaveAddress)
//
// Routine Description:
//
//    Read SFP Slave Address
//
// Arguments:
//    Instance :API instance
//    SlaveAddress : Pointer to the Salve address
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_ReadSlaveAddress(mlbertapi* instance, int *SlaveAddress);

#pragma region declaration 	
//++
//
//	bool mlBert_SFP_SetSlaveAddress(mlbertapi* instance, int SlaveAddress)
//
// Routine Description:
//
//    Set SFP Slave Address
//
// Arguments:
//    Instance :API instance
//    SlaveAddress : Slave address 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_SetSlaveAddress(mlbertapi* instance, int SlaveAddress);

#pragma region declaration 
//++
//
//	APICALL mlBert_SFP_SetRegister(mlbertapi* instance, UInt16 Register, UInt16 Value)
//
// Routine Description:
//
//    Set SFP Register 
//
// Arguments:
//    Instance :API instance
//    Register : Register address 
//    Value : Value for this register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_SetRegister(mlbertapi* instance, UInt16 Register, UInt16 Value);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_GetRegister(mlbertapi* instance, UInt16 Register, UInt16 * Value)
//
// Routine Description:
//
//    Get SFP Register 
//
// Arguments:
//    Instance :API instance
//    Register : Register address 
//    *Value : Pointer for the Value of this register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetRegister(mlbertapi* instance, UInt16 Register, UInt16 * Value);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_TXFaultStatus(mlbertapi* instance)
//
// Routine Description:
//
//    Check SFP RX Fault status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No TX Fault and true if TX fault 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_TXFaultStatus(mlbertapi* instance);

#pragma region declaration
//++
//
//	bool mlBert_SFP_RXLOSStatus(mlbertapi* instance)
//
// Routine Description:
//
//    Check SFP RX LOS Status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RXLOSStatus(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_TXDisable(mlbertapi* instance, int Mode)
//
// Routine Description:
//
//    Disable TX 
//
// Arguments:
//    Instance :API instance
//    Mode = 0 to enable TX and 1 to Disable TX
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_TXDisable(mlbertapi* instance, int Mode);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_ModulePresent(mlbertapi* instance)
//
// Routine Description:
//
//    Check SFP Module Present status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if module not present and true if module present
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_ModulePresent(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS0RateSelect(mlbertapi* instance, int mode)
//
// Routine Description:
//
//    Select SFP RS0 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS0RateSelect(mlbertapi* instance, int mode);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS1RateSelect(mlbertapi* instance, int mode)
//
// Routine Description:
//
//    Select SFP RS1 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS1RateSelect(mlbertapi* instance, int mode);

#pragma region declaration 
//++
//
//	APICALL bool mlBert_SFP_GetTemperature(mlbertapi* instance, double* Temperature)
//
// Routine Description:
//
//    Read SFP Module Temperature
//
// Arguments:
//    Instance :API instance
//    Temperature : Temperature Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetTemperature(mlbertapi* instance, double* Temperature);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_GetVolt_sens(mlbertapi* instance, double* Voltage)
//
// Routine Description:
//
//    Read SFP Module Voltage level
//
// Arguments:
//    Instance :API instance
//    Voltage : Voltage Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetVolt_sens(mlbertapi* instance, double* Voltage);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_GetCurrent_sens(mlbertapi* instance, double* Current)
//
// Routine Description:
//
//    Read SFP Module Current level
//
// Arguments:
//    Instance :API instance
//    Current : Current Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetCurrent_sens(mlbertapi* instance, double* Current);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_ModulePresent(mlbertapi* instance)
//
// Routine Description:
//
//    Check SFP Module Present status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if module not present and true if module present
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_ModulePresent(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS0RateSelect(mlbertapi* instance, int mode)
//
// Routine Description:
//
//    Select SFP RS0 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS0RateSelect(mlbertapi* instance, int mode);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_RS1RateSelect(mlbertapi* instance, int mode)
//
// Routine Description:
//
//    Select SFP RS1 Mode
//
// Arguments:
//    Instance :API instance
//    mode : 1 hight mode 0 low mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_RS1RateSelect(mlbertapi* instance, int mode);


#pragma region declaration 
//++
//
//	APICALL bool mlBert_SFP_GetTemperature(mlbertapi* instance, double* Temperature)
//
// Routine Description:
//
//    Read SFP Module Temperature
//
// Arguments:
//    Instance :API instance
//    Temperature : Temperature Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetTemperature(mlbertapi* instance, double* Temperature);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_GetVolt_sens(mlbertapi* instance, double* Voltage)
//
// Routine Description:
//
//    Read SFP Module Voltage level
//
// Arguments:
//    Instance :API instance
//    Voltage : Voltage Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetVolt_sens(mlbertapi* instance, double* Voltage);

#pragma region declaration 
//++
//
//	bool mlBert_SFP_GetCurrent_sens(mlbertapi* instance, double* Current)
//
// Routine Description:
//
//    Read SFP Module Current level
//
// Arguments:
//    Instance :API instance
//    Current : Current Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SFP_GetCurrent_sens(mlbertapi* instance, double* Current);


#pragma region declaration 
//++
//
// int mlBert_GetEyeScale(mlbertapi* instance, int Channel, double *Margin)
//
// Routine Description:
// 
// this function is used to read the Eye Y scale also it read the min and max y value in steps 
//
// Arguments:
//
// instance: API instance
// Channel: the Channel number
// Margin: convertion value
//
#pragma endregion
APICALL void  STACKMODE mlBert_GetEyeScale(mlbertapi* instance, int Channel, double *Margin);

#pragma region declaration 
//++
//
// int mlBert_SetAttenuator(mlbertapi* instance, int Channel, double Attenuator_dB)
//
// Routine Description:
// 
// this function is used to set the RX Channel Attenuator value in dB
//
// Arguments:
//
// instance: API instance
// Channel: the Channel number
// Attenuator_dB: Attenuator value in dB
//
#pragma endregion
APICALL void  STACKMODE mlBert_SetAttenuator(mlbertapi* instance, int Channel, double Attenuator_dB);

#pragma region declaration
//++
//
//	int mlBert_GetPhaseShiftValue(mlbertapi* instance, int channel)
//
// Routine Description:
//
//    Read the Phase Shfit for a specific channel
//
// Arguments:
//
//    instance: API instance
//    Channel:  channel Number
//
// Return Value:
//
//    Returns the phase shift value
//
//--
#pragma endregion
APICALL double  STACKMODE mlBert_GetPhaseShiftValue(mlbertapi* instance, int channel);

#pragma region 407 Fucntions

#pragma region declaration 
//++
//
//	int mlBert_SetVCOFrequency(mlbertapi* instance, double VCOFrequency)
//
// Routine Description:
//
//    Set VCO Frequency output
//
// Arguments:
//
//    instance : API instance
//    VCOFrequency: The VCO frequncy desired (range 2.43Ghz to 2.585Ghz)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_SetVCOFrequency(mlbertapi* instance, double VCOFrequency);

#pragma region declaration
//++
//
//	int mlBert_Enable_DisableFM(mlbertapi* instance, int FM_Status)
//
// Routine Description:
//
//    Enable FM modulation
//
// Arguments:
//
//    instance : API instance
//    FM_Status:  0 to disable and 1 to enable
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_Enable_DisableFM(mlbertapi* instance, int FM_Status);

#pragma region declaration
//++
//
//	int mlBert_SetFMFrequency(mlbertapi* instance, double FM_Frequency)
//
// Routine Description:
//
//    Set Fm Frequency 
//
// Arguments:
//
//    instance : API instance
//    FM_Frequency:  FM Frequency Value (Uint in Mhz)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_SetFMFrequency(mlbertapi* instance, double FM_Frequency);

#pragma region declaration
//++
//
//	int mlBert_SetFMAmplitude(mlbertapi* instance, int FM_Amplitdue)
//
// Routine Description:
//
//    Set Fm Amplitude 
//
// Arguments:
//
//    instance : API instance
//    FM_Amplitdue:  FM Amplitude Value (0 to 4095)
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_SetFMAmplitude(mlbertapi* instance, int FM_Amplitdue);

#pragma region declaration 
//++
//
//	int mlBert_ClockSource_LoadDefaultCalibration(mlbertapi* instance)
//
// Routine Description:
//
//    Load Fm Calibration 
//
// Arguments:
//
//  instance : API instance
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_ClockSource_LoadDefaultCalibration(mlbertapi* instance);

#pragma region declaration 
//++
//
//	int mlBert_SaveFMCalibrationValues(mlbertapi* instance, double Version, double *Data)
//
// Routine Description:
//
//    Save FM Calibration value 
//
// Arguments:
//
//    instance : API instance
//    Version  : FM calibration version
//    Data     : Data to be saved
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_SaveFMCalibrationValues(mlbertapi* instance, double Version, double *Data);

#pragma region declaration 
//++
//
//	int mlBert_EnableFMFrequencyCalibration(mlbertapi* instance, double *MinMax)
//
// Routine Description:
//
//    Enable FM Calibration Value
//
// Arguments:
//
//    instance : API instance
//    MinMax:  array that contain all the min and max value for each FM calibration frequency
// Return Value:
//
//    Returns FM calibration version number
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_EnableFMFrequencyCalibration(mlbertapi* instance, double *MinMax);

#pragma region declaration 
//++
//
//	int mlBert_DisableFMFrequencyCalibration(mlbertapi* instance)
//
// Routine Description:
//
//    Disable FM Calibration Value
//
// Arguments:
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_DisableFMFrequencyCalibration(mlbertapi* instance);

#pragma region declaration 
//++
//
//	int mlBert_ReadVCOLock(mlbertapi* instance)
//
// Routine Description:
//
//    Read the VCO Lock
//
// Arguments:
//
//    instance : API instance
//
// Return Value:
//
//    Returns 1 locked, 0 not locked
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_ReadVCOLock(mlbertapi* instance);


#pragma endregion

#pragma region declaration 
//++
//
//	void mlBert_EnableBUJ(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
//    Enable Disable BUJ
//
// Arguments:
//	  status:   1 for enable, 0 for disable
//    Channel:  channel Number
//--
#pragma endregion
APICALL void   STACKMODE mlBert_EnableBUJ(mlbertapi* instance, int channel, int status);

#pragma region declaration 
//++
//
//	void mlBert_SetBUJfreq(mlbertapi* instance, int channel, int frequency)
//
// Routine Description:
//    Set the BUJ frequency for a specific channel
//
// Arguments:
//	  
//    Channel:  channel Number
//	  frequency: 1-->5G  2 --> 2.5G  3 --> 1.25G  4 --> 625M
//--
#pragma endregion
APICALL void   STACKMODE mlBert_SetBUJfreq(mlbertapi* instance, int channel, int frequency);

#pragma region declaration 
//++
//
//	void mlBert_SetBUJamplitude(mlbertapi* instance, int channel, int amplitude, int index)
//
// Routine Description:
//    Set the BUJ amplitude for a specific channel
//
// Arguments:
//	  
//    Channel:  channel Number
//	  amplitude: 0 to 4095
//--
#pragma endregion
APICALL void   STACKMODE mlBert_SetBUJamplitude(mlbertapi* instance, int channel, int amplitude, int index);

#pragma region declaration 
//++
//
//	void mlBert_OptimizedBUJ(mlbertapi* instance, int channel, double *MinMax)
//
// Routine Description:
//    Optimize BUJ amplitude for a specific channel
//
// Arguments:
//	  
//    Channel:  channel Number
//	  MinMax
//--
#pragma endregion
APICALL int   STACKMODE mlBert_OptimizedBUJ(mlbertapi* instance, int channel, double *MinMax);

#pragma region declaration 
//++
//
//	void mlBert_UnOptimizedBUJ(mlbertapi* instance, int Channel)
//
// Routine Description:
//    UnOptimize BUJ amplitude for a specific channel
//
// Arguments:
//	  
//    Channel:  channel Number	  
//--
#pragma endregion
APICALL int   STACKMODE mlBert_UnOptimizedBUJ(mlbertapi* instance, int Channel);

#pragma region declaration 
//++
//
//	void mlBert_SaveBUJCalibrationValues(mlbertapi* instance, int Channel, double Version, double *Data)
//
// Routine Description:
//    Save BUJ Calibration Values
//
// Arguments:
//	  
//    Channel:  channel Number	
//	  Version
//	  Data
//--
#pragma endregion
APICALL int   STACKMODE mlBert_SaveBUJCalibrationValues(mlbertapi* instance, int Channel, double Version, double *Data);

#pragma region declaration 
//++
//
//	APICALL bool mlBert_XFP_SetRegister(mlbertapi* instance, UInt16 Register, UInt16 Value)
//
// Routine Description:
//
//    Set XFP Register 
//
// Arguments:
//    Instance :API instance
//    Register : Register address 
//    Value : Value for this register
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool   STACKMODE mlBert_XFP_SetRegister(mlbertapi* instance, UInt16 Register, UInt16 Value);
#pragma region declaration 
//++
//
//	bool mlBert_XFP_GetRegister(mlbertapi* instance, UInt16 Register, UInt16 *Value)
//
// Routine Description:
//
//    Read a specfic register from a XFP Module
//
// Arguments:
//
//    instance :API instance
//    Register : contain the register value
//    Value : pointer that contain the Register data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool   STACKMODE mlBert_XFP_GetRegister(mlbertapi* instance, UInt16 Register, UInt16 *Value);

APICALL bool   STACKMODE mlBert_XFP_Interrupt(mlbertapi* instance, UInt16 data);

#pragma region declaration
//++
//
//	bool mlBert_XFP_RXLOSStatus(mlbertapi* instance)
//
// Routine Description:
//
//    Check XFP RX LOS Status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_XFP_RXLOSStatus(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_XFP_TXFaultStatus(mlbertapi* instance)
//
// Routine Description:
//
//    Check XFP RX Fault status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if No TX Fault and true if TX fault 
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_XFP_TXFaultStatus(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_XFP_ModulePresent(mlbertapi* instance)
//
// Routine Description:
//
//    Check XFP Module Present status
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//   return false if module not present and true if module present
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_XFP_ModulePresent(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_XFP_TXDisable(mlbertapi* instance, int Mode)
//
// Routine Description:
//
//    Disable TX 
//
// Arguments:
//    Instance :API instance
//    Mode = 0 to enable TX and 1 to Disable TX
// Return Value:
//
//   return false if No RX LOS and true if RX LOS 
//
//--
#pragma endregion
APICALL bool   STACKMODE mlBert_XFP_TXDisable(mlbertapi* instance, int Mode);

#pragma region declaration 
//++
//
//	bool mlBert_XFP_ModuleSelection(mlbertapi* instance, int Mode)
//
// Routine Description:
//
//    Select module for I2C communication
//
// Arguments:
//
//    instance :API instance
//    Mode : 0 or 1 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int   STACKMODE mlBert_XFP_ModuleSelection(mlbertapi* instance, int Mode);

APICALL int   STACKMODE mlBert_XFP_PowerDown(mlbertapi* instance, int Mode);

#pragma region declaration 
//++
//
//	bool mlBert_XFP_GetVolt_sens(mlbertapi* instance, int mode, double* Voltage)
//
// Routine Description:
//
//    Read XFP Module Voltage level
//
// Arguments:
//    Instance :API instance
//    Voltage : Voltage Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool   STACKMODE mlBert_XFP_GetVolt_sens(mlbertapi* instance, int mode, double* Voltage);

#pragma region declaration 
//++
//
//	bool mlBert_SetAvagoRxConnectionStatus(mlbertapi* instance, int channel, bool RxSingleEnded)
//
// Routine Description:
//
//    Set Avago Rx Connection Status
//
// Arguments:
//    Instance :API instance
//	  channel
//    RxSingleEnded 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void   STACKMODE mlBert_SetAvagoRxConnectionStatus(mlbertapi* instance, int channel, bool RxSingleEnded);

#pragma region declaration 
//++
//
//	bool mlBert_SetAvagoCTLEvalue(mlbertapi* instance, int channel, double linerate)
//
// Routine Description:
//
//    Set Avago CTLE value
//
// Arguments:
//    Instance :API instance
//	  channel
//    linerate 
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void   STACKMODE mlBert_SetAvagoCTLEvalue(mlbertapi* instance, int channel, double linerate);

#pragma region declaration 
//++
//
//	void mlBert_CoarseTuning(mlbertapi* instance, int channel)
//
// Routine Description:
//
//    Coarse Tuning
//
// Arguments:
//    Instance :API instance
//	  channel
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_CoarseTuning(mlbertapi* instance, int channel);

#pragma region declaration 
//++
//
//	void mlBert_HaltTuning(mlbertapi* instance, int channel)
//
// Routine Description:
//
//    Halt Tuning
//
// Arguments:
//    Instance :API instance
//	  channel
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_HaltTuning(mlbertapi* instance, int channel);

#pragma region declaration 
//++
//
//	void mlBert_FineTuning(mlbertapi* instance, int channel)
//
// Routine Description:
//
//    Fine Tuning
//
// Arguments:
//    Instance :API instance
//	  channel
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_FineTuning(mlbertapi* instance, int channel);

#pragma region declaration 
//++
//
//	void mlBert_ContinuousTuning(mlbertapi* instance, int channel)
//
// Routine Description:
//
//    Continuous Tuning
//
// Arguments:
//    Instance :API instance
//	  channel
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_ContinuousTuning(mlbertapi* instance, int channel);

#pragma region declaration 
//++
//
//	void mlBert_FPGA_Write_PatternLength(mlbertapi* instance, int channel, UInt16 Data)
//
// Routine Description:
//
//    to write FPGA PatternLength
//
// Arguments:
//    Instance :API instance
//	  channel
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_FPGA_Write_PatternLength(mlbertapi* instance, int channel, UInt16 Data);

#pragma region declaration 
//++
//
//	void mlBert_FPGA_Read_PatternLength(mlbertapi* instance, int channel, UInt16 Data)
//
// Routine Description:
//
//    to read FPGA PatternLength
//
// Arguments:
//    Instance :API instance
//	  channel
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int STACKMODE mlBert_FPGA_Read_PatternLength(mlbertapi* instance, int channel, UInt16 *Data);

#pragma region declaration 
//++
//
//	int  mlBert_FPGA_Pattern_Data(mlbertapi* instance, int channel, int mode, UInt64 *Pattern)
//
// Routine Description:
//
//    to set FPGA Pattern data
//
// Arguments:
//    Instance :API instance
//	  channel
//	  mode
//    Pattern
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int STACKMODE mlBert_FPGA_Pattern_Data(mlbertapi* instance, int channel, int mode, UInt64 *Pattern);

#pragma region declaration 
//++
//
//	int mlBert_FPGA_Clk_Output_Divider(mlbertapi* instance, int channel, int mode, UInt16 *Data)
//
// Routine Description:
//
//    to set FPGA Clk Output Divider
//
// Arguments:
//    Instance :API instance
//	  channel
//    mode
//	  Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int STACKMODE mlBert_FPGA_Clk_Output_Divider(mlbertapi* instance, int channel, int mode, UInt16 *Data);

#pragma region declaration 
//++
//
//	void mlBert_Enable_Deep_System_Loopback_GB2(mlbertapi* instance)
//
// Routine Description:
//
//    to Enable Deep System Loopback for GB2
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_Enable_Deep_System_Loopback_GB2(mlbertapi* instance);

#pragma region declaration 
//++
//
//	void mlBert_Disable_Deep_System_Loopback_GB2(mlbertapi* instance)
//
// Routine Description:
//
//    to disable Deep System Loopback for GB2
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void STACKMODE mlBert_Disable_Deep_System_Loopback_GB2(mlbertapi* instance);

#pragma region declaration 
//++
//
//	bool mlBert_SetPAM4UserDefinedPattern(mlbertapi* instance, int channel, char* pattern, MLUserDefinedFileFormat fileFormat)
//
// Routine Description:
//
//    to Set PAM4 UserDefined Pattern for 4004PAM
//
// Arguments:
//    Instance :API instance
//	  channel
//    pattern
//    fileFormat
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool STACKMODE mlBert_SetPAM4UserDefinedPattern(mlbertapi* instance, int channel, char* pattern, MLUserDefinedFileFormat fileFormat);

#pragma region declaration 
//++
//
//	bool mlBert_SetVCOFrequencyIn(mlbertapi* instance, double VCOFrequencyIn)
//
// Routine Description:
//
//    to Set VCO frequency Input
//
// Arguments:
//    Instance :API instance
//	  VCOFrequencyIn
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool STACKMODE mlBert_SetVCOFrequencyIn(mlbertapi* instance, double VCOFrequencyIn);

#pragma region declaration 
//++
//
//	bool mlBert_QuickSettings(mlbertapi* instance, byte Mode, byte GrayMapping, byte PreCoding, byte FECenable, byte FECmode, UInt8 Rate, byte Clock, UInt8 TXPattern, UInt8 RXPattern, UInt8 CML, UInt16 Amplitude, UInt8 InnerAmplitude, UInt8 Pre, UInt8 Post, byte Side)
//
// Routine Description:
//
//    to Set settings in one packet for 4039PAM-ATE
//
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool STACKMODE mlBert_QuickSettings(mlbertapi* instance, byte Mode, byte GrayMapping, byte PreCoding, byte FECenable, byte FECmode, UInt8 Rate, byte Clock, UInt8 TXPattern, UInt8 RXPattern, UInt8 CML, UInt16 Amplitude, UInt8 InnerAmplitude, UInt8 Pre, UInt8 Post, byte Side);



#pragma region declaration 
//++
//
//	bool mlBert_CalculateDeep_EyeWidth(mlbertapi* instance, int channel, double targetBER, double* EyeWidth)
//
// Routine Description:
//
//    to Calculate Deep EyeWidth for 4039
//
// Arguments:
//    Instance :API instance
//	  channel
//	  targetBER
//	  EyeWidth
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_CalculateDeep_EyeWidth(mlbertapi* instance, int channel, double targetBER, double* EyeWidth);

#pragma region declaration 
//++
//
//	bool mlBert_CalculateDeep_eyeHeight(mlbertapi* instance, int channel, double targetBER, double* EyeHeight)
//
// Routine Description:
//
//    to Calculate Deep EyeHeight for 4039
//
// Arguments:
//    Instance :API instance
//	  channel
//	  targetBER
//	  EyeHeight
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_CalculateDeep_eyeHeight(mlbertapi* instance, int channel, double targetBER, double* EyeHeight);

#pragma region declaration
//++
//
//	int  mlBert_LineRateConfiguration_Polarise(mlbertapi* instance, double linerate, MLEyeMode_ML4004 EyeMode, int GrayMapping, int PRECoding, int VGAtracking, int clockSource, int ClockType, int divider, int FEC, int FECmode, int IEEEmode, int AFETRIM, int is7taps)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4039D and 4079D
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate
//	clockSource: 1 for internal, 0 for external
//	EyeMode: 0 for PAM mode,1 for NRZ mode
//	PRECoding : 1 to enable, 0 to disable
//	GrayMapping: 1 to enable, 0 to disable
//  VGAtracking : 1
//  ClockType: 0-monitor clk, 1-external clk, 2-reference clk
//  divider: 0 for 4, 1 for 8, 2 for 16, 3 for 32, 4 for 128 
//	FECEnable: 1 to enable, 0 to disable
//  FECmode 
//  IEEEmode: 1
//  AFETRIM: 0 below 500mV, 1 above 500mV
//  is7taps: 1 to enable, 0 to disable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_Polarise(mlbertapi* instance, double linerate, MLEyeMode_ML4004 EyeMode, int GrayMapping, int PRECoding, int VGAtracking, int clockSource, int ClockType, int divider, int FEC, int FECmode, int IEEEmode, int AFETRIM, int is7taps);

#pragma region declaration 
//++
//
//	bool mlBert_OutterEyeLevel(mlbertapi* instance, int channel, int value)
//
// Routine Description:
//
//    to set outer eye level for 4039D/79D/39E/39B
//
// Arguments:
//    Instance :API instance
//	  channel
//	  value: from 1500 to 2500
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_OutterEyeLevel(mlbertapi* instance, int channel, int value);

#pragma region declaration 
//++
//
//	bool mlBert_MainTap(mlbertapi* instance, int channel, int value, int mode)
//
// Routine Description:
//
//    to set main tap for 4039D/79D/39E/39B
//
// Arguments:
//    Instance :API instance
//	  channel
//	  value: from -1000 to 1000
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_MainTap(mlbertapi* instance, int channel, int value, int mode);

#pragma region declaration 
//++
//
//	bool mlBert_ErrorInsertionPolarise(mlbertapi* instance, int channel, int enable, int gap, int mode, int duration)
//
// Routine Description:
//
//    to set error insertion for 4039D/79D/39E/39B
//
// Arguments:
//    Instance :API instance
//	  channel
//    enable: 1
//    gap
//    mode
//	  duration
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ErrorInsertionPolarise(mlbertapi* instance, int channel, int enable, int gap, int mode, int duration);

#pragma region declaration 
//++
//
//	bool mlBert_VPeakEnable(mlbertapi* instance, int channel, int status)
//
// Routine Description:
//
//    to enable vpeak for 4039D/79D
//
// Arguments:
//    Instance :API instance
//	  channel
//    status: 1 to enable, 0 to disable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_VPeakEnable(mlbertapi* instance, int channel, int status);

APICALL bool  STACKMODE mlBert_ChangeIPConfiguration(mlbertapi* instance, char* IP);

APICALL bool  STACKMODE mlBert_SetEthIPMask(mlbertapi* instance, char* ipMask);

APICALL bool  STACKMODE mlBert_SetEthIPGateway(mlbertapi* instance, char* ipGateway);

APICALL bool  STACKMODE mlBert_readIPConfiguration(mlbertapi* instance, char * IP);

APICALL bool  STACKMODE mlBert_readIPMask(mlbertapi* instance, char * IPMask);

APICALL bool  STACKMODE mlBert_readGateway(mlbertapi* instance, char * IPGateway);

#pragma region declaration 
//++
//
//	bool mlBert_QDD_MODPRS(mlbertapi* instance, int Module, bool* status)
//
// Routine Description:
//
//    Check if the QDD module is present 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    status : Pointer that containe the Module Status (true or false)
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_MODPRS(mlbertapi* instance, int Module, bool* status);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_LPMODE(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    enable diable QDD low power mode 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means set low power mode on, false disable low power mode
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_LPMODE(mlbertapi* instance, int Module, bool asserted);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_RESET(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    reset QDD Module 
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means Disable QSFP, false means enable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_RESET(mlbertapi* instance, int Module, bool asserted);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_MODSEL(mlbertapi* instance, int Module, bool asserted)
//
// Routine Description:
//
//    Select module for I2C communication
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means I2C communication is selectd for this module,
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_MODSEL(mlbertapi* instance, int Module, bool asserted);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_IntL(mlbertapi* instance, int Module, bool* status)
//
// Routine Description:
//
//    Check I2c Interput  
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    asserted : if true means I2C interput is ON
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_IntL(mlbertapi* instance, int Module, bool* status);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_Get_VCCTX(mlbertapi* instance, int Module, double *Data)
//
// Routine Description:
//
//    Read Vcc TX Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_Get_VCCTX(mlbertapi* instance, int Module, double *Data);
#pragma region declaration
//++
//
//	bool mlBert_QDD_Get_VCCRX(mlbertapi* instance, int Module, double *Data)
//
// Routine Description:
//
//    Read Vcc RX Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_Get_VCCRX(mlbertapi* instance, int Module, double *Data);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_GetVCC(mlbertapi* instance, int Module, double* Data)
//
// Routine Description:
//
//    Read Vcc Voltage Level
//
// Arguments:
//
//    instance :API instance
//    Module : 0 or 1 to select the QSFP Module
//    Data : Contain the valotage level
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_GetVCC(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetVCC1(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetVCCTX(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetVCCRX(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetCurrentVCC(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetCurrentVCCTX(mlbertapi* instance, int Module, double *Data);

APICALL bool  STACKMODE mlBert_QDD_GetCurrentVCCRX(mlbertapi* instance, int Module, double *Data);

#pragma region declaration 
//++
//
//	bool mlBert_QDD_GetVolt_sens(mlbertapi* instance, int module, double* Voltage)
//
// Routine Description:
//
//    Read QDD Module Voltage level
//
// Arguments:
//    Instance :API instance
//    Voltage : Voltage Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_GetVolt_sens(mlbertapi* instance, int module, double* Voltage);
#pragma region declaration 
//++
//
//	bool mlBert_QDD_GetCurrent_sens(mlbertapi* instance, double* Current)
//
// Routine Description:
//
//    Read QDD Module Current level
//
// Arguments:
//    Instance :API instance
//    Current : Current Pointer
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_QDD_GetCurrent_sens(mlbertapi* instance, int module, double* Current);

APICALL bool  STACKMODE mlBert_QDD_PageSelect(mlbertapi* instance, int module, int Page);

APICALL bool  STACKMODE mlBert_QDD_BankSelect(mlbertapi* instance, int module, int Bank);

APICALL bool  STACKMODE mlBert_QDD_ReadI2C(mlbertapi* instance, int module, int address, int Page, int Bank, byte* Data);

APICALL bool  STACKMODE mlBert_QDD_WriteI2C(mlbertapi* instance, int module, int address, int Page, int Bank, byte Data);

APICALL bool  STACKMODE mlBert_QDD_ReadI2CBlock(mlbertapi* instance, int module, int address, int Page, int Bank, int Length, byte* Data);

APICALL bool  STACKMODE mlBert_QDD_WriteI2CBlock(mlbertapi* instance, int module, int address, int Page, int Bank, int Length, byte* Data);

#pragma region declaration 
//++
//
//	bool mlBert_WriteDoubleToEEProm(mlbertapi* instance, UInt32 Address, double Data)
//
// Routine Description:
//
//    write double values to EEPROM
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_WriteDoubleToEEProm(mlbertapi* instance, UInt32 Address, double Data);
#pragma region declaration 
//++
//
//	bool mlBert_ReadDoubleFromEEPROM(mlbertapi* instance, UInt32 Address, double *Data)
//
// Routine Description:
//
//    read double values from EEPROM
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ReadDoubleFromEEPROM(mlbertapi* instance, UInt32 Address, double *Data);



#pragma region declaration
//++
//
//	int  mlBert_LineRateConfiguration_Porrima(mlbertapi* instance, double linerate, MLEyeMode_ML4004 EyeMode, int GrayMapping, int PRECoding, int ChipMode, int clockSource, int ClockType, int divider, int FEC, int FECmode, int IEEEmode, int alltaps)
//
// Routine Description:
//
//	Configure Board Line rate for Ml4039E
//
// Arguments:
//
//  instance :API instance
//  lineRate: Contain the line rate
//             required in Gbps
//			  NB: * The clock file should be included in the specified directory
//                  in ConfigureApplication:saveConfig parameter (default:clk)
//                * Configuration will be lost after applying Linerate
//	clockSource: 1 for internal, 0 for external
//	EyeMode: 0 for PAM mode,1 for NRZ mode
//	PRECoding : 1 to enable, 0 to disable
//	GrayMapping: 1 to enable, 0 to disable
//  ChipMode : 1
//  ClockType: 0-monitor clk, 1-external clk, 2-reference clk
//  divider: 0 for 4, 1 for 8, 2 for 16, 3 for 32, 4 for 128 
//	FECEnable: 1 to enable, 0 to disable
//  FECmode 
//  IEEEmode: 1
//  alltaps: 1 to enable, 0 to disable
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
#pragma endregion
APICALL int  STACKMODE mlBert_LineRateConfiguration_Porrima(mlbertapi* instance, double linerate, MLEyeMode_ML4004 EyeMode, int GrayMapping, int PRECoding, int ChipMode, int clockSource, int ClockType, int divider, int FEC, int FECmode, int IEEEmode, int alltaps);
#pragma region declaration 
//++
//
//	bool mlBert_WriteValueToEEPROM(mlbertapi* instance, UINT16 Address, UINT16 Data)
//
// Routine Description:
//
//    write value to EEPROM
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_WriteValueToEEPROM(mlbertapi* instance, UINT16 Address, UINT16 Data);
#pragma region declaration 
//++
//
//	bool mlBert_ReadValueFromEEPROM(mlbertapi* instance, UINT16 Address, UINT16 *Data)
//
// Routine Description:
//
//    read value from EEPROM
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_ReadValueFromEEPROM(mlbertapi* instance, UINT16 Address, UINT16 *Data);
#pragma region declaration 
//++
//
//	bool mlBert_SaveOptimalSettings(mlbertapi* instance, int Channel, int mode, double *Data)
//
// Routine Description:
//
//    save optimal settings for calibrated mode
//
// Arguments:
//    Instance :API instance
//    Channel
//    mode
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_SaveOptimalSettings(mlbertapi* instance, int Channel, int mode, double *Data);
#pragma region declaration 
//++
//
//	bool mlBert_ReadOptimalSettings(mlbertapi* instance, int Channel, int mode, double *Data)
//
// Routine Description:
//
//    read optimal settings for calibrated mode
//
// Arguments:
//    Instance :API instance
//    Channel
//    mode
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_ReadOptimalSettings(mlbertapi* instance, int Channel, int mode, double *Data);
#pragma region declaration 
//++
//
//	bool  mlBert_ReadEEPROMValues(mlbertapi* instance, int Address, double *Data)
//
// Routine Description:
//
//    read all values from EEPROM
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_ReadEEPROMValues(mlbertapi* instance, int Address, double *Data);
#pragma region declaration 
//++
//
//	bool mlBert_ReadTXLock(mlbertapi* instance, int channel, bool *status)
//
// Routine Description:
//
//    read tx lock status
//
// Arguments:
//    Instance :API instance
//    channel
//    status
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL bool  STACKMODE mlBert_ReadTXLock(mlbertapi* instance, int channel, bool *status);
#pragma region declaration 
//++
//
//	bool mlBert_GetFFE(mlbertapi* instance, int Channel, double *taps)
//
// Routine Description:
//
//   get FFE taps for 4039E
//
// Arguments:
//    Instance :API instance
//    channel
//    taps
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_GetFFE(mlbertapi* instance, int Channel, double *taps);
#pragma region declaration 
//++
//
//	bool mlBert_ReadBerMSBLSB(mlbertapi* instance, int channel, double* BERMSB, double* BERLSB)
//
// Routine Description:
//
//   Read BER MSB and LSB for 4039D/79D
//
// Arguments:
//    Instance :API instance
//    channel
//    BERMSB
//    BERLSB
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ReadBerMSBLSB(mlbertapi* instance, int channel, double* BERMSB, double* BERLSB);
#pragma region declaration 
//++
//
//	bool mlBert_ReadErrorCorrectionPolarisAll(mlbertapi* instance, UInt64 errorCorrected[], UInt64 BlockCount[], UInt64 SaturatedBlockCount[], UInt64 ErrorCount[], UInt64* TimeSpent)
//
// Routine Description:
//
//   Read Error corrected, block count and saturated block count, when using FEC for 4079D
//
// Arguments:
//    Instance :API instance
//    errorCorrected
//    BlockCount
//    SaturatedBlockCount
//    ErrorCount
//    TimeSpent
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ReadErrorCorrectionPolarisAll(mlbertapi* instance, UInt64 errorCorrected[], UInt64 BlockCount[], UInt64 SaturatedBlockCount[], UInt64 ErrorCount[], UInt64* TimeSpent);
#pragma region declaration 
//++
//
//	bool mlBert_WriteEEPROMAddressSecondPage(mlbertapi* instance, UInt32 Address, double Data)
//
// Routine Description:
//
//   write to EEPROM second page
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int STACKMODE mlBert_WriteEEPROMAddressSecondPage(mlbertapi* instance, UInt32 Address, double Data);
#pragma region declaration 
//++
//
//	bool mlBert_ReadEEPROMValuesSecondPage(mlbertapi* instance, UInt32 Address, double* Data)
//
// Routine Description:
//
//   read from EEPROM second page
//
// Arguments:
//    Instance :API instance
//    Address
//    Data
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ReadEEPROMValuesSecondPage(mlbertapi* instance, UInt32 Address, double* Data);
#pragma region declaration 
//++
//
//	bool mlBert_SetSecondary(mlbertapi* instance)
//
// Routine Description:
//
//   set secondary wrapper used for 4039EN
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL void  STACKMODE mlBert_SetSecondary(mlbertapi* instance);
#pragma region declaration 
//++
//
//	bool mlBert_ReadSER(mlbertapi* instance, UInt64 Symbols[])
//
// Routine Description:
//
//   get SER taps for 4079D/39D
//
// Arguments:
//    Instance :API instance
//    Symbols
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
#pragma endregion
APICALL int  STACKMODE mlBert_ReadSER(mlbertapi* instance, UInt64 Symbols[]);

APICALL void  STACKMODE mlBert_Configuration_Stream(mlbertapi* instance, UInt16 *address, UInt16 *data, UInt16 length);

APICALL int  STACKMODE mlBert_Silab_Stream(mlbertapi* instance, UInt16 *address, UInt16 *data, UInt16 length);

APICALL void  STACKMODE mlBert_ReadChannelStatus(mlbertapi* instance, int Channel, double *data);

APICALL void  STACKMODE mlBert_ReadOptimalSettingsSream(mlbertapi* instance, int Channel, double *data);

APICALL void  STACKMODE mlBert_ReadOldCalValues(mlbertapi* instance, int Channel, double *data);

//++
//
//	bool mlBert_Relock(mlbertapi* instance)
//
// Routine Description:
//
//   Apply RX relock control, implemented in favor of ML4039E
//
// Arguments:
//    Instance :API instance
// Return Value:
//
//    Returns if the command was executed properly Success = 1, failed = 0
//
//--
// 
APICALL bool STACKMODE mlBert_Relock(mlbertapi* instance);

#endif